/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/*eslint-env browser*/

/**
 * API Status: **Private**
 * @module nmodule/cloudSentienceConnector/rc/DeviceRegistrationWidget
 */
define([ 'baja!',
    'bajaux/Widget',
    'bajaux/mixin/subscriberMixIn',
    'lex!cloudSentienceConnector',
    'Promise',
    'jquery',
    'underscore',
    'dialogs',
    'hbs!nmodule/cloudSentienceConnector/rc/DeviceRegistrationWidgetTemplate',
    'hbs!nmodule/cloudSentienceConnector/rc/DeviceRegistrationDlgTemplate',
    'css!nmodule/cloudSentienceConnector/rc/DeviceRegistrationWidgetStyle' ], function (
    baja,
    Widget,
    subscriberMixIn,
    lexicons,
    Promise,
    $,
    _,
    dialogs,
    devRegTemplate,
    devRegDlgTemplate) {

    'use strict';

    var lex = lexicons[0],
        successImg = "<img src='/module/icons/x16/job/success.png'/>",
        failureImg = "<img src='/module/icons/x16/job/failed.png'/>",
        pendingImg = "<img class='devRegSpin' src='/module/icons/x16/refresh.png'/>";


    /**
     * A Widget that enables a device to be registered with the Cloud.
     *
     * @class
     * @alias module:nmodule/cloudSentienceConnector/rc/DeviceRegistrationWidget
     * @extends module:bajaux/Widget
     */
    var DeviceRegistrationWidget = function () {
        var that = this;
        Widget.apply(that, arguments);

        subscriberMixIn(that);
    };

    DeviceRegistrationWidget.prototype = new Widget();
    DeviceRegistrationWidget.prototype.constructor = DeviceRegistrationWidget;

    DeviceRegistrationWidget.prototype.doLoad = function (srv) {
        var that = this,
            impl = srv.getConnectorImpl();

        function callExternalResource(url) {
            return impl.rpc("getBajaVersion")
                .then(function (bajaVersion) {
                    var query = "";

                    function addParam(name, val) {
                        query += (query ? "&" : "?") + name + "=" + encodeURIComponent(val);
                    }

                    addParam("systemId", impl.getSystemId());
                    addParam("systemType", impl.getSystemType());
                    addParam("systemOwnershipCode", impl.getSystemOwnershipCode());
                    addParam("systemPublicKey", impl.getSystemPublicKey());
                    addParam("systemRegistrationState", impl.getRegistrationState().getTag());
                    if (srv.getId()) {
                        addParam("systemGuid", srv.getId());
                    }
                    addParam("bajaVersion", bajaVersion);
                    if (impl.getGatewayId() !== "") {
                        addParam("gatewayId", impl.getGatewayId());
                    }


                    url += query;

                    // Seems to be some re-encoding issues with URLs and ORDs so workaround this for now.
                    if (window.niagara.env.type === "wb") {
                        window.niagara.env.hyperlink(url);
                    } else {
                        window.open(url, "_blank");
                    }
                });
        }

        function reconnect() {
            return srv.rpc("reconnect");
        }

        // When focus is put on the window, attempt a reconnect if we're currently disconnected.
        // The system has to be registered and the connector enabled, otherwise reconnect() is guaranteed to fail.
        $(window).on("focus", (that.$reconnect = function () {
            if (impl.getRegistrationState().is("registered") &&
                srv.getEnabled() &&
                srv.getConnectionState().is("disconnected")) {
                reconnect();
            }
        }));

        return that.getSubscriber().subscribe(impl)
            .then(function () {
                var jq = that.jq(),
                    regStatusJq,
                    currentRegStatus,
                    connEnabledJq,
                    currentConnEnabled,
                    connStateJq,
                    currentConnStat,
                    systemTypeJq,
                    systemGuidJq,
                    gatewayIdJq,
                    registerButtonJq,
                    migrationButtonJq,
                    reconnectButtonJq,
                    errorSpanJq;

                jq.html(devRegTemplate({
                    title: lex.get("DeviceRegistrationWidget.title"),
                    deviceRegistered: lex.get("DeviceRegistrationWidget.deviceRegistered"),
                    connectorEnabled: lex.get("DeviceRegistrationWidget.connectorEnabled"),
                    deviceConnected: lex.get("DeviceRegistrationWidget.deviceConnected"),
                    systemType: lex.get("DeviceRegistrationWidget.systemType"),
                    systemGuid: lex.get("DeviceRegistrationWidget.systemGuid"),
                    registerDevice: lex.get("DeviceRegistrationWidget.registerDevice"),
                    forceReconnect: lex.get("DeviceRegistrationWidget.forceReconnect"),
                    gatewayId: lex.get("DeviceRegistrationWidget.gatewayId"),
                    migrateDevice: lex.get("DeviceRegistrationWidget.migrateDevice")
                }));

                jq.on("click", ".devRegRegister", function (e) {
                    e.preventDefault();

                    if (impl.getRegistrationState().is("registered") && srv.getConnectionState().is("connected")) {
                        dialogs.showYesNo(lex.get("DeviceRegistrationWidget.alreadyRegistered"))
                            .yes(callExternalResource(impl.getDeviceRegistrationUrl()));
                        // TODO: This should either not be possible, or should reset the connector first, because
                        // otherwise it WILL fail, if we actually attempt re-registration.
                    } else {
                        callExternalResource(impl.getDeviceRegistrationUrl());
                    }
                });

                jq.on("click", ".devRegReconnect", function (e) {
                    e.preventDefault();
                    reconnect();
                });

                jq.on("click", ".devMigrateBtn", function (e) {
                    e.preventDefault();
                    callExternalResource(impl.getMigrationUrl());
                });

                regStatusJq = jq.find(".devRegStatus");
                connEnabledJq = jq.find(".connEnabled");
                connStateJq = jq.find(".devRegConnState");
                systemTypeJq = jq.find(".devRegSystemType");
                gatewayIdJq = jq.find(".devRegGatewayId");
                systemGuidJq = jq.find(".devRegSystemGuid");
                registerButtonJq = jq.find(".devRegRegister");
                reconnectButtonJq = jq.find(".devRegReconnect");
                migrationButtonJq = jq.find(".devMigrateBtn");
                errorSpanJq = jq.find(".devRegError");

                function updateConnectionDetails() {
                    var newRegStatus,
                        newConnEnabled,
                        newConnState;

                    // This should show failure for both unregistered and registered but not provisioned states
                    newRegStatus = impl.getRegistrationState().is("registered") ? successImg : failureImg;

                    if (impl.getGatewayId() === '') {
                        gatewayIdJq.parent().hide();
                    } else {
                        gatewayIdJq.text(impl.getGatewayId());
                    }

                    if (newRegStatus !== currentRegStatus) {
                        regStatusJq.html((currentRegStatus = newRegStatus));
                    }

                    if (srv.getEnabled()) {
                        newConnEnabled = successImg;
                    } else {
                        newConnEnabled = failureImg;
                    }

                    if (newConnEnabled !== currentConnEnabled) {
                        connEnabledJq.html((currentConnEnabled = newConnEnabled));
                    }

                    if (srv.getConnectionState().is("connected")) {
                        newConnState = successImg;
                    } else if (srv.getConnectionState().is("disconnected")) {
                        newConnState = failureImg;
                    } else {
                        newConnState = pendingImg;
                    }

                    if (newConnState !== currentConnStat) {
                        connStateJq.html((currentConnStat = newConnState));
                    }

                    systemTypeJq.text(impl.getSystemType() || lex.get("DeviceRegistrationWidget.none"));
                    systemGuidJq.text(srv.getIdDisplay() || lex.get("DeviceRegistrationWidget.none"));

                    if (impl.getMigrationUrl() === '') {
                        migrationButtonJq.hide();
                    }

                    // Disable the "Register Device" button when registration is guaranteed to fail:
                    // - missing system type
                    // - invalid certificate
                    if (!impl.getSystemType() || impl.getSystemType().len === 0) {
                        registerButtonJq.attr("disabled", "disabled");
                        errorSpanJq.text(lex.get("DeviceRegistrationWidget.missingSystemType"));
                    } else if (!impl.getCertIsValid() && impl.getCloudCertVerification()) {
                        registerButtonJq.attr("disabled", "disabled");
                        errorSpanJq.text(lex.get("DeviceRegistrationWidget.badCert"));
                    } else {
                        registerButtonJq.removeAttr("disabled");
                        errorSpanJq.text("");
                    }

                    // Disable the "Reconnect" button when we know beforehand that
                    // there is no point in trying to connect:  not registered or not enabled
                    if (!impl.getRegistrationState().is("registered") || !srv.getEnabled()) {
                        reconnectButtonJq.attr("disabled", "disabled");
                    } else {
                        reconnectButtonJq.removeAttr("disabled");
                    }
                }

                updateConnectionDetails();
                that.getSubscriber().attach("changed", _.throttle(updateConnectionDetails, 100));

                // If the service is currently registered and enabled, but disconnected, then attempt to reconnect.
                if (impl.getRegistrationState().is("registered") &&
                    srv.getEnabled() &&
                    srv.getConnectionState().is("disconnected")) {
                    // If we detect the registration part has failed then ask the user to register the device.
                    that.getSubscriber().attach("topicFired", _.once(function (topic) {
                        if (topic.getName() === "registrationFailed") {
                            dialogs.showYesNo(lex.get("DeviceRegistrationWidget.forward"))
                                .yes(callExternalResource(impl.getDeviceRegistrationUrl()));
                        }
                    }));

                    reconnect();
                }
            });
    };

    DeviceRegistrationWidget.prototype.doDestroy = function () {
        if (this.$reconnect) {
            $(window).off("focus", this.$reconnect);
        }
    };

    return DeviceRegistrationWidget;
});
