/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/baja/BacnetObject
 */
define(['baja!', 'underscore', 'nmodule/bacnet/rc/baja/enums/BacnetObjectType', 'nmodule/bacnet/rc/baja/util/BacnetBitStringUtil', 'nmodule/webEditors/rc/baja/bajaUtils', 'nmodule/webEditors/rc/fe/baja/util/compUtils'], function (baja, _, BacnetObjectType, BacnetBitStringUtil, bajaUtils, compUtils) {
  'use strict';

  var Component = baja.Component,
    getBitStringFacets = BacnetBitStringUtil.getBitStringFacets,
    proxySlotFacets = bajaUtils.proxySlotFacets,
    getClosest = compUtils.closest;
  var generateBitStringFacets = _.once(function () {
    return {
      statusFlags: getBitStringFacets('BacnetStatusFlags'),
      eventEnable: getBitStringFacets('BacnetEventTransitionBits'),
      ackedTransitions: getBitStringFacets('BacnetEventTransitionBits'),
      limitEnable: getBitStringFacets('BacnetLimitEnable')
    };
  });
  function getObjectTypeFacets(device) {
    return device.get('enumerationList').get('objectTypeFacets');
  }

  /**
   * BajaScript implementation of a `bacnet:BacnetObject`.
   *
   * @class
   * @extends baja.Component
   * @alias module:nmodule/bacnet/rc/baja/BacnetObject
   */
  var BacnetObject = function BacnetObject() {
    Component.apply(this, arguments);
    proxySlotFacets(this, generateBitStringFacets());
  };
  BacnetObject.prototype = Object.create(Component.prototype);
  BacnetObject.prototype.constructor = BacnetObject;

  /**
   * Find the `bacnet:BacnetDevice` for this object - should be the parent
   * of the containing `bacnet:BacnetConfigDeviceExt`.
   *
   * @returns {baja.Component|null}
   */
  BacnetObject.prototype.device = function () {
    var ext = getClosest(this, 'bacnet:BacnetConfigDeviceExt');
    if (ext) {
      var dev = ext.getParent();
      if (baja.hasType(dev, 'bacnet:BacnetDevice')) {
        return dev;
      }
    }
    return null;
  };

  /**
   * For `objectId` and `objectType`, get the facets from the device's object
   * type facets.
   *
   * `statusFlags`, `eventEnable`, `ackedTransitions`, and `limitEnable` get
   * bit string facets pulled from the `bacnet` lexicon.
   *
   * @see BBacnetObject#getSlotFacets(Slot)
   * @param {String|baja.Slot} slot
   * @returns {baja.Facets}
   */
  BacnetObject.prototype.getFacets = function (slot) {
    var f, device;
    switch (String(slot)) {
      case 'objectId':
        if (this.isMounted()) {
          f = BacnetObjectType.getObjectIdFacets(this.get('objectType').getOrdinal());
          if (f) {
            return f;
          }
          device = this.device();
          if (device) {
            return getObjectTypeFacets(device);
          }
        }
        break;
      case 'objectType':
        device = this.device();
        if (device) {
          return getObjectTypeFacets(device);
        }
    }
    return Component.prototype.getFacets.apply(this, arguments);
  };
  return BacnetObject;
});
