/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/baja/datatypes/BacnetOctetString
 */
define(['baja!', 'underscore'], function (baja, _) {
  'use strict';

  var Simple = baja.Simple;
  function byteToHexString(_byte) {
    _byte = (_byte || 0) & 0xff;
    var s = _byte.toString(16);
    if (s.length === 1) {
      s = '0' + s;
    }
    return s;
  }
  function bytesToHexString(bytes, separator) {
    bytes = bytes || [];
    return _.map(bytes, byteToHexString).join(separator);
  }
  function parseHexByte(byteStr) {
    return parseInt(byteStr, 16);
  }

  /**
   * BajaScript representation of an `bacnet:BacnetOctetString` value.
   *
   * @class
   * @extends {baja.Simple}
   * @alias {module:nmodule/bacnet/rc/baja/datatypes/BacnetOctetString}
   */
  var BacnetOctetString = function BacnetOctetString(bytes) {
    Simple.apply(this, arguments);
    this.$bytes = bytes || [];
  };
  BacnetOctetString.prototype = Object.create(Simple.prototype);
  BacnetOctetString.prototype.constructor = BacnetOctetString;

  /**
   * Factory method to create a new `BacnetOctetString` instance from an array of bytes
   * @static
   * @param {Array} bytes Array of bytes
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/BacnetOctetString}
   */
  BacnetOctetString.make = function (bytes) {
    return new BacnetOctetString(bytes);
  };

  /**
   * Instance method to create a new `BacnetOctetString` instance from an array of bytes
   * @param bytes
   * @param {Array} bytes Array of bytes
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/BacnetOctetString}
   */
  BacnetOctetString.prototype.make = function (bytes) {
    return BacnetOctetString.make(bytes);
  };

  /**
   * Encode the simple using a String format
   * that can be translated using decodeFromString.
   * @returns {String} String encoding of byte array
   */
  BacnetOctetString.prototype.encodeToString = function () {
    if (!this.$bytes.length) {
      return 'null';
    }
    return bytesToHexString(this.$bytes, ' ');
  };

  /**
   *
   * Decode the simple using the same String format
   * that was written using encodeToString, and return
   * the new instance.  Under no circumstances should
   * this instance be modified.
   *
   * @param str
   */
  BacnetOctetString.prototype.decodeFromString = function (str) {
    if (!str || str === 'null') {
      return BacnetOctetString.DEFAULT;
    }
    var bytes = _.map(str.split(' '), parseHexByte);
    return new BacnetOctetString(bytes);
  };

  /**
   * Returns the Array of bytes representing the octet string
   * @returns {Array} Array of bytes
   */
  BacnetOctetString.prototype.getBytes = function () {
    return _.clone(this.$bytes);
  };

  /**
   * `DEFAULT` is initialized to an empty byte array
   * @type {module:nmodule/bacnet/rc/baja/datatypes/BacnetOctetString}
   */
  BacnetOctetString.DEFAULT = new BacnetOctetString();
  baja.registerType('bacnet:BacnetOctetString', function () {
    return BacnetOctetString;
  });
  return BacnetOctetString;
});
