/**
 * @copyright 2016, Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/fe/MacAddressEditor
 */

define(['baja!', 'baja!bacnet:BacnetOctetString', 'Promise', 'nmodule/webEditors/rc/fe/baja/StringEditor', 'nmodule/bacnet/rc/baja/datatypes/BacnetAddress', 'nmodule/bacnet/rc/baja/datatypes/BacnetOctetString'], function (baja, types, Promise, StringEditor, BacnetAddress, BacnetOctetString) {
  'use strict';

  var stringToBytes = BacnetAddress.stringToBytes,
    bytesToString = BacnetAddress.bytesToString;

  /**
   * <p>MacAddressEditor is a field editor for a bacnet:BacnetOctetString type</p>
   *
   * <b>Note</b>: This editor relies on an <b>'addressType'</b> property being pre-initialized
   * in its properties (see BacnetAddressEditor's makeBuilder() method for an example).
   * The 'addressType' property corresponds to the specific MAC Address style constant
   * used to load/read this editor (e.g BacnetAddress.MAC_TYPE_ETHERNET)
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/StringEditor
   * @alias module:nmodule/bacnet/rc/fe/MacAddressEditor
   * @see nmodule/bacnet/rc/fe/BacnetAddressEditor
   * @see nmodule/bacnet/rc/baja/BacnetAddress
   */

  var MacAddressEditor = function MacAddressEditor() {
    StringEditor.apply(this, arguments);
  };

  //extend and set up prototype chain
  MacAddressEditor.prototype = Object.create(StringEditor.prototype);
  MacAddressEditor.prototype.constructor = MacAddressEditor;

  /**
   * Returns the string representation of the loaded MAC address
   * @returns {String} MAC Address
   */
  MacAddressEditor.prototype.$getMacAddress = function () {
    return this.$getTextElement().val();
  };

  /**
   * Loads the editor with a value of type 'bacnet:BacnetOctetString
   *
   * @param value instance of type bacnet:BacnetOctetString to load
   * @returns {Promise} Promise resolved when editor is loaded or rejected when value
   *                   of an incorrect type is loaded
   */
  MacAddressEditor.prototype.doLoad = function (octetString) {
    var bytes, addressType, str;
    if (!baja.hasType(octetString, 'bacnet:BacnetOctetString')) {
      return Promise.reject(new Error('Expected type bacnet:BacnetOctetString'));
    }
    bytes = octetString.getBytes();
    addressType = this.properties().getValue('addressType');
    str = addressType !== null ? bytesToString(addressType, bytes) : '';
    return StringEditor.prototype.doLoad.call(this, str);
  };

  /**
   * Reads the current MAC Address string encoding and returns a new bacnet:BacnetOctetString instance
   * based on the 'addressType' property the editor was originally configured with.
   *
   * @returns {Promise} Promise resolved with a new instance of a bacnet:BacnetOctetString
   */
  MacAddressEditor.prototype.doRead = function () {
    var addressType = this.properties().getValue('addressType'),
      macStr = this.$getMacAddress(),
      isIp,
      hasIpPort,
      len,
      bytes;
    if (addressType !== null) {
      isIp = addressType === BacnetAddress.MAC_TYPE_IP;
      hasIpPort = isIp && macStr.indexOf(":") !== -1;
      len = hasIpPort ? 5 : isIp ? 4 : 0;
      bytes = stringToBytes(addressType, len, macStr);
      return BacnetOctetString.make(bytes);
    }
    return BacnetOctetString.make();
  };
  return MacAddressEditor;
});
