/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/wb/mgr/BacnetPointUxManager
 */
define(['baja!', 'Promise', 'underscore', 'dialogs', 'nmodule/bacnet/rc/baja/enums/BacnetPropertyIdentifier', 'nmodule/bacnet/rc/wb/mgr/model/BacnetPointLearnModel', 'nmodule/bacnet/rc/wb/mgr/model/BacnetPointManagerModel', 'nmodule/bacnet/rc/util/BacnetConst', 'nmodule/bacnet/rc/util/ObjectTypeList', 'nmodule/bacnet/rc/util/PropertyInfo', 'nmodule/driver/rc/wb/mgr/PointMgr', 'nmodule/driver/rc/wb/mgr/PointMgrModel', 'nmodule/webEditors/rc/fe/baja/util/compUtils', 'nmodule/webEditors/rc/wb/mgr/MgrTypeInfo', 'nmodule/webEditors/rc/wb/mgr/MgrLearn', 'nmodule/webEditors/rc/wb/tree/TreeNode', 'css!nmodule/bacnet/rc/bacnet', 'baja!bacnet:BacnetPointFolder,' + 'bacnet:BacnetObjectType,' + 'bacnet:BacnetProxyExt,' + 'bacnet:DiscoveryPoint'], function (baja, Promise, _, dialogs, BacnetPropertyIdentifier, BacnetPointLearnModel, BacnetPointManagerModel, BacnetConst, ObjectTypeList, PropertyInfo, PointMgr, PointMgrModel, compUtils, MgrTypeInfo, addMgrLearnSupport, TreeNode) {
  'use strict';

  var NUMERIC_ICON = baja.Icon.make(['module://icons/x16/control/numericPoint.png']),
    STRING_ICON = baja.Icon.make(['module://icons/x16/control/stringPoint.png']),
    BOOLEAN_ICON = baja.Icon.make(['module://icons/x16/control/booleanPoint.png']),
    ENUM_ICON = baja.Icon.make(['module://icons/x16/control/enumPoint.png']),
    OBJECT_ICON = baja.Icon.make(['module://bacnet/com/tridium/bacnet/ui/icons/bacObject.png']),
    CONTROL_POINT_TYPE = baja.lt('control:ControlPoint'),
    PROXY_EXT_TYPE = baja.lt('bacnet:BacnetProxyExt'),
    BACNET_POINT_FOLDER_TYPE = baja.lt('bacnet:BacnetPointFolder'),
    OBJECT_TYPE_ENUM = baja.$('bacnet:BacnetObjectType'),
    ANALOG_OUTPUT = OBJECT_TYPE_ENUM.get('analogOutput'),
    ANALOG_VALUE = OBJECT_TYPE_ENUM.get('analogValue'),
    BINARY_OUTPUT = OBJECT_TYPE_ENUM.get('binaryOutput'),
    BINARY_VALUE = OBJECT_TYPE_ENUM.get('binaryValue'),
    MULTI_STATE_OUTPUT = OBJECT_TYPE_ENUM.get('multiStateOutput'),
    MULTI_STATE_VALUE = OBJECT_TYPE_ENUM.get('multiStateValue'),
    PRIORITIZED_PRESENT_VALUE_FACET = 'priPV',
    LOADING_DIALOG_DELAY_MS = 500;

  /**
   * Create a function that is called at most one time to resolve the default
   * object list. This may take an argument that can specify the ord - this is only
   * used for unit testing purposes, and is specified as a custom parameter to
   * the initialize() call, which should be the first call to this function.
   * This will normally be called without an argument when called internally from
   * other places within this class.
   *
   * The intention here is that it could be called for multiple managers on a view
   * but will only result in a single network call for the file.
   *
   * @param {String} [ord] - an ord to use for the default list; intended for unit testing only.
   * @returns {Promise.<module:nmodule/bacnet/rc/util/ObjectTypeList>}
   */
  var getDefaultObjectTypeList = _.once(function (ord) {
    return ObjectTypeList.make(ord ? {
      ord: ord
    } : undefined);
  });

  ////////////////////////////////////////////////////////////////
  // BacnetPointUxManager
  ////////////////////////////////////////////////////////////////

  /**
   * BacnetPointUxManager constructor. Mixes in learn support.
   *
   * @class
   * @alias module:nmodule/bacnet/rc/wb/mgr/BacnetPointUxManager
   * @extends module:nmodule/driver/rc/wb/mgr/PointMgr
   * @param {Object} params
   */
  var BacnetPointUxManager = function BacnetPointUxManager(params) {
    PointMgr.call(this, _.extend({
      moduleName: 'bacnet',
      keyName: 'BacnetPointUxManager',
      folderType: BACNET_POINT_FOLDER_TYPE,
      subscriptionDepth: 3
    }, params));
    addMgrLearnSupport(this);
  };
  BacnetPointUxManager.prototype = Object.create(PointMgr.prototype);
  BacnetPointUxManager.prototype.constructor = BacnetPointUxManager;

  /**
   * Overrides the device manager's doInitialize() function. This will load
   * the default object type list in parallel with calling the base class
   * initialization.
   *
   * @param {JQuery} dom
   * @param {Object} [params]
   * @returns {*}
   */
  BacnetPointUxManager.prototype.doInitialize = function (dom, params) {
    var that = this,
      objectTypeListOrd = params && params.objectTypeListOrd;
    that.on('jobcomplete', function (job) {
      if (job.getType().is('bacnet:BacnetDiscoverPointsJob')) {
        that.$discoveryJobComplete(job, that);
      }
    });
    return Promise.all([PointMgr.prototype.doInitialize.apply(that, arguments), getDefaultObjectTypeList(objectTypeListOrd)]);
  };

  /**
   * Look at the device to see whether it has a vendor specific object XML file ord
   * specified as a dynamic property. If so, return a `Promise` to resolve an
   * `ObjectTypeList` from that file, and set it as a private property on the manager.
   *
   * @param {module:nmodule/bacnet/rc/wb/mgr/BacnetPointUxManager} mgr - the BACnet point manager
   * @param {baja.Component} device - the BACnet device
   *
   * @returns {Promise}
   */
  function loadVendorObjectTypeList(mgr, device) {
    var vendorXmlOrd;
    vendorXmlOrd = device.get('vendorObjectTypesFile');
    if (vendorXmlOrd) {
      return ObjectTypeList.make({
        ord: vendorXmlOrd
      }).then(function (list) {
        mgr.$vendorObjectTypeList = list;
        return list;
      });
    }
    return Promise.resolve(null);
  }

  /**
   * Load the point device extension or the point folder into the manager.
   * It will attempt to resolve the vendor specific object type list at this point.
   *
   * @returns {Promise}
   */
  BacnetPointUxManager.prototype.doLoad = function (value, params) {
    var that = this,
      device = that.getDevice(),
      promise;

    //
    // Load in the slots on the device first - we want to have the device's
    // 'enumerationList' property and the dynamic vendor object type list
    // properties loaded up front. The BacnetProxyExt type ext makes use of
    // the 'enumerationList' property for facet values.
    //

    promise = device.loadSlots().then(function () {
      return Promise.all([PointMgr.prototype.doLoad.apply(that, [value, params]), loadVendorObjectTypeList(that, device)]);
    });
    dialogs.showLoading(LOADING_DIALOG_DELAY_MS, promise);
    return promise;
  };

  /**
   * Destroy the manager.
   *
   * @returns {Promise.<*>}
   */
  BacnetPointUxManager.prototype.doDestroy = function () {
    if (this.$vendorObjectTypeList) {
      delete this.$vendorObjectTypeList;
    }
    return PointMgr.prototype.doDestroy.apply(this, arguments);
  };

  /**
   * Function to return an array of types to be offered upon the execution
   * of the command to add new instances.
   *
   * @returns {Promise.<Array.<module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo>>}
   */
  BacnetPointUxManager.prototype.getNewTypes = function () {
    return PointMgrModel.getDefaultNewTypes();
  };

  /**
   * Returns the point ext for the current loaded component. As we may have a folder
   * loaded, this may be several levels up in the component tree.
   *
   * @returns {baja.Component}
   */
  BacnetPointUxManager.prototype.getDeviceExt = function () {
    return compUtils.closest(this.value(), baja.lt('bacnet:BacnetPointDeviceExt'));
  };

  /**
   * Override of the `componentChanged` handler. This will inspect the changed
   * property to see if the change can be ignored, so avoid updating the table
   * DOM unecessarily for frequently changing, but perhaps uninteresting properties.
   *
   * @param {baja.Component} component
   * @param {baja.Property} prop
   * @returns {*}
   */
  BacnetPointUxManager.prototype.componentChanged = function (component, prop) {
    var type, readColumn;
    type = component.getType();

    // If the change is from something such as an alarm ext (e.g. time in current state)
    // we can ignore it. We don't want to update the table DOM unnecessarily.

    if (!(type.is(CONTROL_POINT_TYPE) || type.is(PROXY_EXT_TYPE) || type.is(BACNET_POINT_FOLDER_TYPE))) {
      return;
    }

    // Similarly, the 'readValue' property may be updated regularly. Check whether the
    // corresponding column is hidden, and if so, ignore the update.

    if (prop && prop.getName() === 'readValue') {
      readColumn = this.getModel().getColumn('readValue');
      if (readColumn && readColumn.isUnseen()) {
        return;
      }
    }
    return PointMgr.prototype.componentChanged.apply(this, arguments);
  };

  /**
   * Return the device from the current component's point ext.
   *
   * @returns {baja.Component}
   */
  BacnetPointUxManager.prototype.getDevice = function () {
    return this.getDeviceExt().getParent();
  };

  /**
   * Makes the model for the main database table.
   *
   * @param {baja.Component} component - a bacnet network or bacnet device folder instance.
   * @returns {Promise.<module:nmodule/bacnet/rc/wb/mgr/BacnetPointManagerModel>}
   */
  BacnetPointUxManager.prototype.makeModel = function (component) {
    return this.getNewTypes().then(function (newTypes) {
      return new BacnetPointManagerModel({
        component: component,
        newTypes: newTypes
      });
    });
  };

  ////////////////////////////////////////////////////////////////
  // Learn Table
  ////////////////////////////////////////////////////////////////

  function getPoints(table) {
    return table.getSlots().properties().is('bacnet:DiscoveryPoint').toValueArray();
  }
  function getDiscoveryChildren(mgr, discoveryPoint) {
    var promise = mgr.getJob().discover(discoveryPoint).then(function (table) {
      return Promise.all(getPoints(table).map(function (child) {
        return makeDiscoveryTableNode(mgr, child);
      }));
    });
    dialogs.showLoading(LOADING_DIALOG_DELAY_MS, promise);
    return promise;
  }

  /**
   * Return the icon to be used for a tree node in the learn table. This will be
   * based on the ASN type of the discovery point's property id.
   *
   * @param {module:nmodule/bacnet/rc/wb/mgr/BacnetPointUxManager} mgr - the BACnet point manager
   * @param {module:nmodule/bacnet/rc/baja/job/DiscoveryPoint} discovery - a discovered point.
   *
   * @returns {baja.Icon}
   */
  function getIconForDiscoveryPoint(mgr, discovery) {
    var objectType = discovery.getObjectId().getObjectType(),
      propertyId = discovery.getPropertyId();
    return mgr.getPropertyInfo(objectType, propertyId).then(function (propertyInfo) {
      var asnType = propertyInfo.getAsnType();
      switch (asnType) {
        case BacnetConst.ASN_NULL:
          {
            return STRING_ICON;
          }
        case BacnetConst.ASN_BOOLEAN:
          {
            return BOOLEAN_ICON;
          }
        case BacnetConst.ASN_UNSIGNED:
        case BacnetConst.ASN_INTEGER:
          {
            return ENUM_ICON;
          }
        case BacnetConst.ASN_REAL:
        case BacnetConst.ASN_DOUBLE:
          {
            return NUMERIC_ICON;
          }
        case BacnetConst.ASN_OCTECT_STRING:
        case BacnetConst.ASN_CHARACTER_STRING:
        case BacnetConst.ASN_BIT_STRING:
          {
            return STRING_ICON;
          }
        case BacnetConst.ASN_ENUMERATED:
          {
            if (propertyInfo && propertyInfo.getType() === 'bacnet:BacnetBinaryPv') {
              return BOOLEAN_ICON;
            }
            return ENUM_ICON;
          }
        case BacnetConst.ASN_DATE:
        case BacnetConst.ASN_TIME:
        case BacnetConst.ASN_OBJECT_IDENTIFIER:
        case BacnetConst.ASN_CONSTRUCTED_DATA:
        case BacnetConst.ASN_BACNET_ARRAY:
        case BacnetConst.ASN_BACNET_LIST:
        case BacnetConst.ASN_ANY:
        case BacnetConst.ASN_CHOICE:
        case BacnetConst.ASN_UNKNOWN_PROPRIETARY:
          {
            return STRING_ICON;
          }
        default:
          {
            return OBJECT_ICON;
          }
      }
    });
  }

  /**
   * `TreeNode` derived type for the learn table. This has a `loadKids()`
   * override that will invoke an action on the station side to find
   * the child properties of an object as BDiscoveryPoint instances.
   *
   * @param name
   * @param discovery
   * @param mgr
   * @param icon
   * @constructor
   */
  var LearnTableNode = function LearnTableNode(name, discovery, mgr, icon) {
    TreeNode.call(this, name, name);
    this.$discovery = discovery;
    this.$mgr = mgr;
    this.$icon = icon;
  };
  LearnTableNode.prototype = Object.create(TreeNode.prototype);
  LearnTableNode.prototype.constructor = LearnTableNode;

  /**
   * Get the icon for the tree node. This will be looked up according
   * to the data type.
   *
   * @returns {Array.<String>}
   */
  LearnTableNode.prototype.getIcon = function () {
    return this.$icon ? this.$icon.getImageUris() : [];
  };

  /**
   * Load the kids for this discovery node. This involves invoking the
   * 'discover' action on the discovery job.
   *
   * @param {Object} params
   * @returns {Promise.<Array.<module:nmodule/webEditors/rc/wb/tree/TreeNode>>}
   */
  LearnTableNode.prototype.$loadKids = function (params) {
    return getDiscoveryChildren(this.$mgr, this.$discovery).then(function (kids) {
      if (params && params.progressCallback) {
        params.progressCallback('commitReady');
      }
      return kids;
    });
  };

  /**
   * Return the discovery point instance for this node.
   * @returns {module:nmodule/bacnet/rc/baja/job/DiscoveryPoint}
   */
  LearnTableNode.prototype.value = function () {
    return this.$discovery;
  };

  /**
   * Returns true if the data type of the discovery point is one that may
   * have child points.
   *
   * @returns {boolean}
   */
  LearnTableNode.prototype.mayHaveKids = function () {
    return this.$discovery.hasChildren();
  };

  /**
   * Make a tree node for an item in the discovery table.
   *
   * @param {module:nmodule/bacnet/rc/wb/mgr/BacnetPointUxManager} mgr - the BACnet point manager
   * @param {module:nmodule/bacnet/rc/baja/job/DiscoveryPoint} discovery - a discovered point.
   *
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/tree/TreeNode>}
   */
  function makeDiscoveryTableNode(mgr, discovery) {
    var name = discovery.getObjectName();
    return getIconForDiscoveryPoint(mgr, discovery).then(function (icon) {
      return new LearnTableNode(name, discovery, mgr, icon);
    });
  }

  /**
   * Reload the learn table from the manager's discovered points.
   *
   * @param {module:nmodule/bacnet/rc/wb/mgr/BacnetPointUxManager} mgr - the BACnet point manager
   * @returns {Promise}
   */
  function reloadLearnTable(mgr) {
    var model = mgr.getLearnModel();
    return Promise.all((mgr.$discoveries || []).map(function (discovery) {
      return makeDiscoveryTableNode(mgr, discovery);
    })).then(function (nodes) {
      // TODO: create an 'updateLearnRoots' function in the discovery mixin to make this functionality available elsewhere.
      return model.clearRows().then(function () {
        return model.insertRows(nodes, 0);
      });
    });
  }

  /**
   * Get the `PropertyInfo` instance for the given object type and property id.
   * This is similar to the functionality provided by the device class in the Java code.
   * If this is needed elsewhere in the ux module, this can be moved out into a shared
   * file at a later time. This will try to use the vendor specific XML configured on
   * the device first and that's not available, then it will use the generic file.
   *
   * @private
   * @param {number} objectType
   * @param {number} propertyId
   *
   * @returns {Promise.<module:nmodule/bacnet/rc/util/PropertyInfo>}
   */
  BacnetPointUxManager.prototype.getPropertyInfo = function (objectType, propertyId) {
    var propertyInfo = this.getVendorPropertyInfo(objectType, propertyId);
    if (propertyInfo) {
      return Promise.resolve(propertyInfo);
    }
    return getDefaultObjectTypeList().then(function (objectTypeList) {
      propertyInfo = objectTypeList && objectTypeList.getPropertyInfo(objectType, propertyId);
      if (!propertyInfo) {
        propertyInfo = new PropertyInfo({
          name: BacnetPropertyIdentifier.tagForId(propertyId),
          id: propertyId,
          asnType: BacnetConst.ASN_UNKNOWN_PROPRIETARY
        });
      }
      return propertyInfo;
    });
  };

  /**
   * Try to get a `PropertyInfo` instance from the vendor specific XML file, if the file's
   * ord has been declared as a dynamic property on the device.
   *
   * @private
   * @param {number} objectType
   * @param {number} propertyId
   *
   * @returns {module:nmodule/bacnet/rc/util/PropertyInfo} a `PropertyInfo` or null if one could not be obtained
   */
  BacnetPointUxManager.prototype.getVendorPropertyInfo = function (objectType, propertyId) {
    return this.$vendorObjectTypeList ? this.$vendorObjectTypeList.getPropertyInfo(objectType, propertyId) : null;
  };

  /**
   * Make the model for the discovery table. In most cases, the column's value
   * will be a property on a BDiscoveryDevice struct.
   *
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/table/tree/TreeTableModel>}
   */
  BacnetPointUxManager.prototype.makeLearnModel = function () {
    return BacnetPointLearnModel.make();
  };

  ////////////////////////////////////////////////////////////////
  // Discovery
  ////////////////////////////////////////////////////////////////

  /**
   * Resolve the discovery job's ord, and set it on the manager. This is used
   * when clicking on the discover command button, and also when restoring the
   * manager state from a prior discovery.
   *
   * @private
   * @param {String} ord - the ord of a submitted discovery job
   */
  BacnetPointUxManager.prototype.$setDiscoveryJobFromOrd = function (ord) {
    var that = this,
      job;
    return baja.Ord.make(ord).get().then(function (resolved) {
      job = resolved;
      return that.setJob({
        jobOrOrd: job,
        depth: 3
      });
    }).then(function () {
      return job;
    });
  };

  /**
   * Called when the discovery job completes, this will get the discovered
   * items from the job and reload the table.
   *
   * @private
   * @param {baja.Component} job - the network discovery job.
   */
  BacnetPointUxManager.prototype.$discoveryJobComplete = function (job) {
    var that = this;
    job.loadSlots().then(function () {
      var props = job.getSlots().properties();
      that.$discoveries = props.is('bacnet:DiscoveryPoint').toValueArray();
      return reloadLearnTable(that);
    })["catch"](baja.error);
  };

  /**
   * Start the discovery work. This will pop up a dialog with an editor
   * for a DeviceDiscoveryConfig instance. Once the parameters have been
   * set, we can submit the discovery job and set it on the manager.
   *
   * @returns {Promise}
   */
  BacnetPointUxManager.prototype.doDiscover = function () {
    var that = this,
      pointExt = that.getDeviceExt();
    return pointExt.submitPointDiscoveryJob().then(function (ord) {
      ord = baja.Ord.make({
        base: baja.Ord.make('station:'),
        child: ord.relativizeToSession()
      });
      return that.$setDiscoveryJobFromOrd(ord);
    });
  };

  /**
   * Add the two standard boolean point types to the array, optionally
   * with the writable type first.
   */
  function addBooleanTypes(arr, writableFirst) {
    addTypes(arr, writableFirst, 'Boolean');
  }

  /**
   * Add the two standard numeric point types to the array, optionally
   * with the writable type first.
   */
  function addNumericTypes(arr, writableFirst) {
    addTypes(arr, writableFirst, 'Numeric');
  }

  /**
   * Add the two standard enum point types to the array, optionally
   * with the writable type first.
   */
  function addEnumTypes(arr, writableFirst) {
    addTypes(arr, writableFirst, 'Enum');
  }

  /**
   * Add the two standard string point types to the array, optionally
   * with the writable type first.
   */
  function addStringTypes(arr, writableFirst) {
    addTypes(arr, writableFirst, 'String');
  }

  /**
   * Add the type specs for the given point data type to the array.
   */
  function addTypes(arr, writableFirst, prefix) {
    prefix = 'control:' + prefix;
    if (writableFirst) {
      arr.push(prefix + 'Writable', prefix + 'Point');
    } else {
      arr.push(prefix + 'Point', prefix + 'Writable');
    }
  }

  /**
   * Return the types available to be created from a given discovery item.
   * This will look at the BACnet object type to determine whether a writable
   * point should be the preferred first choice.
   *
   * @returns {Promise.<Array.<module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo>>}
   */
  BacnetPointUxManager.prototype.getTypesForDiscoverySubject = function (discovery) {
    var objectType = discovery.getObjectId().getObjectType(),
      propertyId = discovery.getPropertyId();
    return this.getPropertyInfo(objectType, propertyId).then(function (propertyInfo) {
      var types = [],
        asnType = propertyInfo.getAsnType(),
        writableFirst = false;
      if (objectType === ANALOG_OUTPUT.getOrdinal() || objectType === BINARY_OUTPUT.getOrdinal() || objectType === MULTI_STATE_OUTPUT.getOrdinal()) {
        writableFirst = true;
      }
      switch (asnType) {
        case BacnetConst.ASN_NULL:
        case BacnetConst.ASN_BACNET_LIST:
        case BacnetConst.ASN_CONSTRUCTED_DATA:
          {
            return [];
          }
        case BacnetConst.ASN_BOOLEAN:
          {
            addBooleanTypes(types, writableFirst);
            break;
          }
        case BacnetConst.ASN_UNSIGNED:
        case BacnetConst.ASN_INTEGER:
          {
            addEnumTypes(types, writableFirst);
            addBooleanTypes(types, writableFirst);
            break;
          }
        case BacnetConst.ASN_REAL:
        case BacnetConst.ASN_DOUBLE:
          {
            addNumericTypes(types, writableFirst);
            break;
          }
        case BacnetConst.ASN_ENUMERATED:
          {
            if (propertyInfo && propertyInfo.getType() === 'bacnet:BacnetBinaryPv') {
              addBooleanTypes(types, writableFirst);
            }
            addEnumTypes(types, writableFirst);
            break;
          }
        case BacnetConst.ASN_BACNET_ARRAY:
          {
            if (discovery.getPropertyArrayIndex() > 0) {
              if (objectType === ANALOG_OUTPUT.getOrdinal() || objectType === ANALOG_VALUE.getOrdinal()) {
                addNumericTypes(types, true);
                addBooleanTypes(types, true);
                addEnumTypes(types, true);
              } else if (objectType === BINARY_OUTPUT.getOrdinal() || objectType === BINARY_VALUE.getOrdinal()) {
                addBooleanTypes(types, true);
                addEnumTypes(types, true);
                addNumericTypes(types, true);
              } else if (objectType === MULTI_STATE_OUTPUT.getOrdinal() || objectType === MULTI_STATE_VALUE.getOrdinal()) {
                addEnumTypes(types, true);
                addNumericTypes(types, true);
                addBooleanTypes(types, true);
              } else {
                addNumericTypes(types, false);
                addBooleanTypes(types, false);
                addEnumTypes(types, false);
              }
            }
            break;
          }
        default:
          {
            break;
          }
      }
      addStringTypes(types, false);
      return Promise.all(_.map(types, MgrTypeInfo.make));
    });
  };

  /**
   * Test whether the given facets reference is defined and has
   * at least one key.
   *
   * @param {baja.Facets} facets
   * @returns {boolean}
   */
  function hasFacets(facets) {
    return !!(facets && facets.getKeys().length);
  }

  /**
   * Return `true` if the given argument is one of the writable control point types.
   *
   * @param {baja.Component} pt - a `control:ControlPoint` instance.
   * @returns {boolean}
   */
  function isWritablePoint(pt) {
    var type = pt.getType().toString();
    return /^control:.+Writable$/.test(type);
  }

  /**
   * Return the given facets object with the 'prioritized present value'
   * facet removed, if it is present.
   *
   * @param {baja.Facets} facets
   * @returns {baja.Facets}
   */
  function removePrioritizedPVFacet(facets) {
    var key = PRIORITIZED_PRESENT_VALUE_FACET,
      obj = facets.toObject();
    if (obj[key]) {
      delete obj[key];
    }
    return baja.Facets.make(obj);
  }

  /**
   * Returns a `Promise` that will resolve to a `boolean` value that indicates whether
   * the discovered point uses a priority array for writes. This will be used to
   * determine the value of the 'enabled' property on a point created from a discovery
   * item. This may involve a network call to establish whether the object has a priority
   * array.
   *
   * @param {baja.Component} discovery - the discovered item
   * @param {baja.Component} point - the `control:ControlPoint` being configured
   * @param {baja.Component} job - the point discovery job
   *
   * @returns {Promise.<boolean>}
   */
  function checkWritablePrioritized(discovery, point, job) {
    var objectType = discovery.getObjectId().getObjectType();
    if (isWritablePoint(point)) {
      if (objectType === ANALOG_OUTPUT.getOrdinal() || objectType === BINARY_OUTPUT.getOrdinal() || objectType === MULTI_STATE_OUTPUT.getOrdinal()) {
        return Promise.resolve(true);
      } else if (objectType === ANALOG_VALUE.getOrdinal() || objectType === BINARY_VALUE.getOrdinal() || objectType === MULTI_STATE_VALUE.getOrdinal()) {
        return job.checkForPriorityArray(discovery.getObjectId());
      }
    }
    return Promise.resolve(false);
  }

  /**
   * Return the facets for the given discovery item. If the item has facets in its slot,
   * this will be returned. If not, an action will be invoked to get the facets server
   * side, which may involve property reads on the client device.
   *
   * @param {baja.Component} discovery - a discovered item obtained from the discovery job.
   * @param {module:nmodule/bacnet/rc/baja/datatypes/BacnetObjectIdentifier} objectId - the object id obtained from the discovery point
   * @param {baja.Component} job - the discovery job
   * @returns {Promise.<baja.Facets>}
   */
  function getDiscoveryPointFacets(discovery, objectId, job) {
    var discoveryFacets = discovery.get('facets');
    return hasFacets(discoveryFacets) ? Promise.resolve(discoveryFacets) : job.discoverFacets(objectId);
  }

  /**
   * Set up the facets for the new control point. This is called asynchronously once
   * the promise returned by `getDiscoveryPointFacets()` has resolved. This is used
   * to set two facet values - 1) the `deviceFacets` slot on the proxy ext, and 2)
   * the facets on the control point's slot.
   *
   * @param {baja.Component} point - the new point created for the discovery object.
   * @param {baja.Facets} discoveryFacets - the facets obtained from the discovery point.
   * @param {module:nmodule/bacnet/rc/util/PropertyInfo} propertyInfo - the info on the BACnet property.
   * @param {baja.Component} device - the BACnet device.
   * @param {Object} values - the object containing the proposed values for the new control point.
   */
  function setPointFacetsFromDiscovery(point, discoveryFacets, propertyInfo, device, values) {
    var facets = point.get('facets'),
      range;
    if (propertyInfo) {
      if (propertyInfo.isEnum()) {
        range = baja.$(propertyInfo.getType()).getRange();
        if (propertyInfo.isExtensible()) {
          range = device.getEnumerationList().getEnumRange(propertyInfo.getType());
        }
        if (range) {
          discoveryFacets = baja.Facets.make(discoveryFacets, {
            'range': range
          });
        }
      }
      if (propertyInfo.getFacetControl() === 'all') {
        facets = discoveryFacets;
      } else if (propertyInfo.getFacetControl() === 'units') {
        facets = baja.Facets.make({
          units: discoveryFacets.get('units')
        });
      } else if (propertyInfo.getFacetControl() !== 'no' && hasFacets(discoveryFacets)) {
        facets = discoveryFacets;
      }
    }
    values.deviceFacets = facets;
    facets = removePrioritizedPVFacet(facets);
    values.facets = facets;
  }

  /**
   * Get the values to be set as the proposals on the batch component editor for
   * the rows being added via the `AddCommand`.
   *
   * @param {baja.Struct} discovery - a DiscoveryDevice struct instance.
   * @param {baja.ControlPoint} point when a match is done, it is the database point that we are matching to
   * @returns {Object} an object populated with properties to be used for the new row.
   */
  BacnetPointUxManager.prototype.getProposedValuesFromDiscovery = function (discovery, point) {
    var that = this,
      promise,
      objectId = discovery.getObjectId(),
      propertyId = discovery.getPropertyId();
    promise = that.getPropertyInfo(objectId.getObjectType(), propertyId).then(function (propertyInfo) {
      var index = discovery.getIndex(),
        presentValueId = baja.$('bacnet:BacnetPropertyIdentifier').get('presentValue'),
        pointName = discovery.getObjectName() || String(discovery.getObjectId()),
        device = that.getDevice(),
        job = that.getJob(),
        proposals;
      if (propertyId !== presentValueId.getOrdinal()) {
        pointName += '-' + discovery.getPropertyIdentifier();
      }
      if (index.length) {
        pointName += '_' + index;
      }
      proposals = {
        name: point && point.getName() ? point.getName() : pointName.replace(/\//g, '.'),
        values: {
          objectId: discovery.getObjectId(),
          propertyId: baja.DynamicEnum.make({
            ordinal: propertyId,
            range: presentValueId.getRange()
          }),
          readStatus: 'unsubscribed',
          writeStatus: 'readonly'
        }
      };
      if (index) {
        proposals.values.propertyArrayIndex = parseInt(index);
      }
      if (propertyInfo) {
        proposals.values.dataType = String(propertyInfo.getDataType());
      }
      if (isWritablePoint(point)) {
        proposals.values.writeStatus = 'writable';
      }
      return checkWritablePrioritized(discovery, point, job).then(function (writablePrioritized) {
        if (writablePrioritized) {
          proposals.values.enabled = false;
        }
        return getDiscoveryPointFacets(discovery, objectId, job);
      }).then(function (discoveryFacets) {
        setPointFacetsFromDiscovery(point, discoveryFacets, propertyInfo, device, proposals.values);
        return proposals;
      });
    });

    // The above promise chain may require property reads from the device, so show
    // the loading dialog if it takes a while.

    dialogs.showLoading(LOADING_DIALOG_DELAY_MS, promise);
    return promise;
  };

  /**
   * Test whether the given object matches the component in the station.
   * The comparison is based on the bacnet object identifier, the property
   * id and the array index.
   *
   * @param discovery
   * @param {baja.Component} comp
   * @returns {boolean} - true if the discovery item and station component match.
   */
  BacnetPointUxManager.prototype.isExisting = function (discovery, comp) {
    var proxyExt, index;
    if (baja.hasType(comp, 'control:ControlPoint')) {
      proxyExt = comp.getProxyExt();
      index = discovery.getIndex() ? parseInt(discovery.getIndex()) : -1;
      return proxyExt.getType().is('bacnet:BacnetProxyExt') && proxyExt.getObjectId().getObjectType() === discovery.getObjectId().getObjectType() && proxyExt.getObjectId().getInstanceNumber() === discovery.getObjectId().getInstanceNumber() && proxyExt.getPropertyId().getTag().equals(discovery.getPropertyIdentifier()) && proxyExt.getPropertyArrayIndex() === index;
    }
    return false;
  };

  ////////////////////////////////////////////////////////////////
  // State
  ////////////////////////////////////////////////////////////////

  /**
   * Save the state we require for restoring after a hyperlink or page reload.
   * In addition to the visible columns saved by the base type, this will save
   * the ord of any job we may be displaying in the job bar and the ord of the
   * point ext.
   *
   * @returns {Object} an object with the state to be persisted
   */
  BacnetPointUxManager.prototype.saveStateForKey = function () {
    var job = this.getJob(),
      jobOrd = job && job.getNavOrd(),
      pointExt = this.getDeviceExt(),
      pointExtOrd = pointExt && pointExt.getNavOrd(),
      obj = {};
    if (pointExtOrd) {
      obj.$pointExtOrd = pointExtOrd.toString();
    }
    if (jobOrd) {
      obj.$jobOrd = jobOrd.toString();
    }
    return obj;
  };

  /**
   * Restore the state we saved in the call to saveStateForOrd(). This
   * will reinstate the learn job from the ord we peristed when the state
   * was saved.
   *
   * @param {Object} obj - an object with data retrieved from the browser storage.
   * @returns {Promise.<*>}
   */
  BacnetPointUxManager.prototype.restoreStateForKey = function (obj) {
    var pointExt = this.getDeviceExt(),
      pointExtOrd = pointExt && pointExt.getNavOrd();
    if (pointExtOrd && obj.$jobOrd && obj.$pointExtOrd) {
      if (pointExtOrd.toString() === obj.$pointExtOrd) {
        // Try to resolve the ord and set the job on the manager. If it
        // can't be resolved (e.g. the job has been cleaned up in the station)
        // then it's not a big deal, just catch the error and carry on. A new
        // discovery job will need to be started again, if required.

        return this.$setDiscoveryJobFromOrd(obj.$jobOrd)["catch"](baja.error);
      }
    }
    return Promise.resolve();
  };
  return BacnetPointUxManager;
});
