/*jshint browser: true */

/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * @private
 * @module nmodule/cloudBackup/rc/EncryptedDistributionView
 */
define(['baja!',
        'lex!',
        'jquery',
        'Promise',
        'bajaux/Widget',
        'dialogs',
        'nmodule/cloudBackup/rc/backupUtils',
        'lex!cloudBackup',
        'hbs!nmodule/cloudBackup/rc/EncryptedDistributionViewTemplate',
        'css!nmodule/cloudBackup/rc/EncryptedDistributionViewStyle'], function (
        baja, 
        lexjs, 
        $,
        Promise,
        Widget,
        dialogs,
        backupUtils,
        lexicons,
        template) {

  'use strict';

  var lex = lexicons[0];

 /**
  * @class
  * @alias module:nmodule/cloudBackup/rc/EncryptedDistributionView
  * @desc Manages Niagara Back ups.
  */
  var EncryptedDistributionView = function () {
    Widget.apply(this, arguments);
  };
  
  EncryptedDistributionView.prototype = Object.create(Widget.prototype);
  EncryptedDistributionView.prototype.constructor = EncryptedDistributionView;

  /**
   * Decrypt a edist file to a dist file.
   * 
   * @param  {jQuery} jq The jQuery wrapper for the widget.
   * @param  eDistFile The encrypted distribution file.
   */
  function decrypt(jq, eDistFile) {
    if (!baja.isOffline() && !backupUtils.isSecure()) {
      return dialogs.showOk(lex.get("decryptWithHttps"));
    }

    return baja.rpc(eDistFile.getNavOrd(), "isDistFileAvailable")
      .then(function (distFileAvailable) {
        return new Promise(function (resolve, reject) {
          if (distFileAvailable) {
            dialogs.showYesNo(lex.get("unencryptedFileFound"))
              .yes(function () {
                resolve(true);
              })
              .no(function () {
                resolve(false);
              });
          }
          else {
            resolve(true);
          }
        });
      })
      .then(function (decrypt) {
        if (!decrypt) {
          return;
        }

        backupUtils.showPassphraseDialog(function (dlg, passphrase) {
          function finishedDecryption(distOrd) {
            if (distOrd) {
              dialogs.showYesNo(lex.get("decryptSuccess"))
                .yes(function () {
                  window.niagara.env.hyperlink(distOrd);
                });
            }
            else {
              dialogs.showOk(lex.get("decryptFail"));
            }
          }  
            
          if (baja.isOffline()) {
            dialogs.showLoading(0, baja.rpc(eDistFile.getNavOrd(), "decryptToDistUsingPassphrase", passphrase)
              .then(finishedDecryption));
          }
          else {
            dialogs.showLoading(0, baja.rpc(eDistFile.getNavOrd(), "getKeyInfo")
              .then(function (ki) {
                return backupUtils.pbkdf2(passphrase, ki.salt, ki.iterationCount, ki.keySize);
              })
              .then(function (key) {
                return baja.rpc(eDistFile.getNavOrd(), "decryptToDistUsingKey", key);
              }).
              then(finishedDecryption));
          }
        });
      });
  }

  /**
   * Load the encrypted distribution view.
   *
   * @param eDistFile The encrypted distribution file.
   * @returns {Promise}
   */
  EncryptedDistributionView.prototype.doLoad = function (eDistFile) {
    var that = this;

    that.jq().html(template({
      ord: String(eDistFile.getNavOrd()),
      message: lex.get("decryptDistFile")
    }));

    that.jq().on("click", ".encrypted-dist-view-inner > button", function () {
      decrypt(that.jq(), eDistFile);
      return false;
    });
  };

  EncryptedDistributionView.prototype.doLayout = function() {
    var parent = this.jq().parent();

    this.jq().children(".encrypted-dist-view-outer").css({
      width: parent.width(),
      height: parent.height()
    });
  };

  return EncryptedDistributionView;
});