/*jshint browser: true */

/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * @private
 * @module nmodule/cloudBackup/rc/backupUtils
 */
define(['dialogs', 
        'lex!cloudBackup', 
        'jquery', 
        'Promise',
        'hbs!nmodule/cloudBackup/rc/passphraseTemplate'], function (
        dialogs, 
        lexicons, 
        $,
        Promise,
        passphraseTemplate) {
  "use strict";
        
  var lex = lexicons[0], 
      authPath = "/login/core/auth.min.js";

  return {
    /**
     * Return true if the browser using https.
     *
     * @private
     * 
     * @returns {Boolean} Returns true if https is being used.
     */
    isSecure: function () {
      return /^https:/i.test(window.location.href);
    },

    /**
     * Show a passphrase dialog, extract the passphrase value and show it to the user.
     *
     * @private
     * 
     * @param  {function} okHandler Called with the dialog and passphrase argument.
     * @returns The dialog instance. 
     */
    showPassphraseDialog: function(okHandler) {
      var dlg = dialogs.showOkCancel({
        title: lex.get("enterPassphrase.title"),
        content: passphraseTemplate({ message: lex.get("enterPassphrase.description") })
      })
      .ok(function (dlg) {
        okHandler(dlg, dlg.content().find("input").val());
      });

      dlg.content().find("input").focus();
      return dlg;
    },

    /**
     * Returns a promise that resolves to an encrypted key.
     *
     * @private
     * 
     * @param  {String} passphrase     The passphrase to be used for encryption.
     * @param  {String} salt           The salt encoded as a base 64 string.
     * @param  {Number} iterationCount The iteration count used for encryption.
     * @param  {Number} keySize        The key size (in bits) used for encryption.
     * @returns {Promise} A promise that's resolved with the key once computed.
     */
    pbkdf2: function (passphrase, salt, iterationCount, keySize) {
      return new Promise(function (resolve, reject) {
        // This is a hack for getting the ScramShaClient JS loading in RequireJS...
        define("sjcl", [], function () {});
        define("UNorm", [], function () {});
        define("sendHttp", [], function () {});
        define("sendHttpHeader", [], function () {});

        require([authPath], function () {
          var sjcl = window.sjcl;
          sjcl.misc.pbkdf2Async({
              password: passphrase, 
              salt: sjcl.codec.base64.toBits(salt), 
              count: iterationCount, 
              length: keySize, 
              hash: 0
            }, 
            function (err, result) {
              if (err) {
                reject(err);
              }
              else {
                resolve(sjcl.codec.base64.fromBits(result));
              }  
            });
        }, reject);
      });
    }
  };        
});
