/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

define(['baja!', 'baja!bql:DynamicTimeRange,bql:FilterSet,bql:FilterEntry', 'nmodule/webEditors/rc/util/PatternFilter'], function (baja, types, PatternFilter) {
  "use strict";

  /**
   * API Status: **Private**
   *
   * A set of utility functions for the history module.
   *
   * @exports nmodule/history/rc/historyUtil
   */
  var historyUtil = {};

  /**
   * Determine whether the following DynamicTimeRange includes now or not
   * @param {baja.Simple} dynamicTimeRange `bql:DynamicTimeRange`
   * @returns {boolean}
   */
  historyUtil.isLive = function (dynamicTimeRange) {
    if (!dynamicTimeRange) {
      return true;
    }
    var encoding = dynamicTimeRange.encodeToString();
    if (encoding.indexOf("timeRange:") === 0) {
      if (encoding.indexOf("endTime") < 0) {
        return true; //no end
      }
      var endEncoding = encoding.split('endTime=')[1];
      var endTime = baja.AbsTime.DEFAULT.decodeFromString(endEncoding);
      return endTime.equals(baja.AbsTime.DEFAULT) || endTime.valueOf() > baja.AbsTime.now().valueOf(); //true if endTime is beyond now
    }
    switch (encoding) {
      case 'today':
      case 'last24Hours':
      case 'weekToDate':
      case 'last7Days':
      case 'monthToDate':
      case 'yearToDate':
        return true;
      case 'yesterday':
      case 'lastWeek':
      case 'lastMonth':
      case 'lastYear':
        return false;
    }
  };

  /**
   * Convert a DynamicTimeRange to a historyScheme paramter String
   * @param {baja.Simple} dynamicTimeRange `bql:DynamicTimeRange`
   * @returns {String}
   */
  historyUtil.toHistoryParameter = function (dynamicTimeRange) {
    var encoding = dynamicTimeRange.encodeToString();
    encoding = encoding.replace(/Time/g, "");
    if (encoding.indexOf("start") !== 0 && encoding.indexOf("end") !== 0) {
      if (encoding.indexOf("timeRange:") === 0) {
        encoding = encoding.substring("timeRange:".length);
      } else {
        encoding = "period=" + encoding;
      }
    }
    return encoding;
  };

  /**
   *
   * @param {baja.Component} filterSet `bql:FilterEntry`
   * @returns {boolean}
   */
  historyUtil.anyActive = function (filterSet) {
    var anyActive = false;
    filterSet.getSlots().properties().is('bql:FilterEntry').each(function (slot) {
      var filterEntry = this.get(slot);
      if (filterEntry.get('active')) {
        anyActive = true;
      }
    });
    return anyActive;
  };

  /**
   * @since Niagara 4.14
   * @param {Array.<string>} systemTags
   * @param {string} systemTagPatterns
   * @returns {boolean} true if any of the given system tags match any of the configured system tag
   * patterns
   * @see BLocalHistoryDatabase#acceptSystemTags
   */
  historyUtil.acceptSystemTags = function (systemTags, systemTagPatterns) {
    if (!systemTags.length) {
      systemTags = [''];
    }
    return PatternFilter.anyMatch(systemTags, PatternFilter.parseList(systemTagPatterns));
  };

  /**
   * @param {baja.NavNode|baja.Component} node a history nav node, or a `history:HistoryConfig`
   * describing a history
   * @param {object} [cx]
   * @param {boolean} [cx.fullHistoryDisplayName] set to true to include the history device name in
   * the display name, if possible
   * @returns {string}
   * @since Niagara 4.14
   */
  historyUtil.getHistoryDisplayName = function (node) {
    var cx = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    if (baja.hasType(node, 'history:HistoryConfig')) {
      var navDisplayName = node.get('navDisplayName');
      // c.f. BFoxHistory
      if (cx.fullHistoryDisplayName) {
        var prefix = baja.SlotPath.unescape(node.get('id').getDeviceName()) + '/';
        return navDisplayName.startsWith(prefix) ? navDisplayName : prefix + navDisplayName;
      } else {
        return navDisplayName;
      }
    }
    return node.getNavDisplayName();
  };
  return historyUtil;
});
