/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author Ashutosh Chaturvedi
 */

/**
 * @module baja/alarm/AlarmArchiveSpace
 */
define(["bajaScript/sys", "bajaScript/baja/nav/NavNodeContainer", "bajaScript/baja/comm/Callback"], function (baja, NavNodeContainer, Callback) {
  "use strict";

  var subclass = baja.subclass,
    callSuper = baja.callSuper,
    objectify = baja.objectify;

  /**
   * Represents a `alarm:AlarmArchiveSpace` in BajaScript.
   *
   * @class
   * @alias module:baja/alarm/AlarmArchiveSpace
   * @extends baja.NavNodeContainer
   */
  var AlarmArchiveSpace = function AlarmArchiveSpace() {
    callSuper(AlarmArchiveSpace, this, [{
      navName: "alarmArchive",
      icon: "module://icons/x16/alarm.png",
      ord: "local:|alarm:archive"
    }]);
  };
  subclass(AlarmArchiveSpace, NavNodeContainer);

  /**
   * Query the alarm archive database for the count of alarm records within the given time range.
   *
   * @param {baja.Simple} timeRange the `bql:DynamicTimeRange` to query
   * @param filterSet
   * @param {Object} [obj] The object literal.
   * @param {Function} [obj.ok] (Deprecated: use Promise) Called once the request has completed.
   * @param {Function} [obj.fail] (Deprecated: use Promise) Called if the request fails.
   * @param {baja.comm.Batch} [obj.batch] if defined, any network calls will be
   * batched into this object.
   *
   * @returns {Promise} A promise that's resolved with the count of alarm records.
   */
  AlarmArchiveSpace.prototype.getRecordCount = function (timeRange, filterSet, obj) {
    obj = objectify(obj);
    var cb = new Callback(obj.ok, obj.fail, obj.batch);
    if (filterSet) {
      baja.comm.getRecordCount({
        t: timeRange.encodeToString(),
        f: baja.bson.encodeValue(filterSet)
      }, cb);
    } else {
      baja.comm.getRecordCount({
        t: timeRange.encodeToString()
      }, cb);
    }
    return cb.promise();
  };

  /**
   * Clear all the records from the alarm database.
   *
   * @param {Object} [obj] The object literal.
   * @param {Function} [obj.ok] (Deprecated: use Promise) Called once the
   * records have been successfully cleared.
   * @param {Function} [obj.fail] (Deprecated: use Promise) Called if the
   * alarm devices fails to resolve.
   * @param {baja.comm.Batch} [obj.batch] if defined, any network calls will be
   * batched into this object.
   *
   * @returns {Promise} A promise that's resolved once all of the records have
   * been cleared.
   */
  AlarmArchiveSpace.prototype.clearAllRecords = function (obj) {
    obj = objectify(obj);
    var cb = new Callback(obj.ok, obj.fail, obj.batch);
    baja.comm.clearAllAlarmRecords({
      a: true
    }, cb);
    return cb.promise();
  };

  /**
   * Clear all records with a timestamp before the specified time.
   * Also update the totalAlarmCount, unackedAlarmCount, openAlarmCount,
   * and inAlarmCount properties all BAlarmClasses.
   *
   * @param {baja.AbsTime} [beforeTime] The earliest time to keep in the result.
   * Records before this time will be removed.
   * @param {Object} [obj] The object literal.
   * @param {Function} [obj.ok] (Deprecated: use Promise) Called once the
   * records have been successfully cleared.
   * @param {Function} [obj.fail] (Deprecated: use Promise) Called if the alarm
   * devices fails to resolve.
   * @param {baja.comm.Batch} [obj.batch] if defined, any network calls will be
   * batched into this object.
   *
   * @returns {Promise} A promise that's resolved once all of the records have
   * been cleared.
   */
  AlarmArchiveSpace.prototype.clearOldRecords = function (beforeTime, obj) {
    obj = objectify(obj);
    var cb = new Callback(obj.ok, obj.fail, obj.batch);
    baja.comm.clearOldAlarmRecords({
      t: beforeTime,
      a: true
    }, cb);
    return cb.promise();
  };

  /**
   * Clear the record with the given uuid.
   * Also update the totalAlarmCount, unackedAlarmCount, openAlarmCount,
   * and inAlarmCount properties all BAlarmClasses.
   *
   * @param {Array} uuids An array of encoded uuids to remove from the database
   * @param {Object} [obj] The object literal.
   * @param {Function} [obj.ok] (Deprecated: use Promise) Called once the
   * records have been successfully cleared.
   * @param {Function} [obj.fail] (Deprecated: use Promise) Called if the alarm
   * devices fails to resolve.
   * @param {baja.comm.Batch} [obj.batch] if defined, any network calls will be
   * batched into this object.
   *
   * @returns {Promise} A promise that's resolved once all of the records have
   * been cleared.
   */
  AlarmArchiveSpace.prototype.clearRecords = function (uuids, obj) {
    obj = objectify(obj);
    var cb = new Callback(obj.ok, obj.fail, obj.batch);
    baja.comm.clearAlarmRecords({
      u: uuids,
      a: true
    }, cb);
    return cb.promise();
  };

  /**
   * Query the alarm database for the given time range
   * @param {baja.Simple} timeRange the `bql:DynamicTimeRange` to query
   * @param limit
   * @param offset
   * @param filterSet
   * @param sortDescending
   * @param columnName
   * @param {Object} [obj] The object literal.
   * @param {Function} [obj.ok] (Deprecated: use Promise) Called once the request has completed.
   * @param {Function} [obj.fail] (Deprecated: use Promise) Called if the request fails.
   * @param {baja.comm.Batch} [obj.batch] if defined, any network calls will be
   * batched into this object.
   *
   * @returns {Promise} A promise that's resolved once all of the records have
   * been retrieved
   */
  AlarmArchiveSpace.prototype.queryAlarmDatabase = function (timeRange, limit, offset, filterSet, sortDescending, columnName, obj) {
    obj = objectify(obj);
    var cb = new Callback(obj.ok, obj.fail, obj.batch);
    if (filterSet) {
      baja.comm.queryAlarmDatabase({
        t: timeRange.encodeToString(),
        l: limit,
        o: offset,
        f: baja.bson.encodeValue(filterSet),
        d: sortDescending,
        c: columnName,
        a: true
      }, cb);
    } else {
      baja.comm.queryAlarmDatabase({
        t: timeRange.encodeToString(),
        l: limit,
        o: offset,
        d: sortDescending,
        c: columnName,
        a: true
      }, cb);
    }
    return cb.promise();
  };

  /**
   * Add a note to an alarm record
   * @param alarm The alarm record to which to add the note
   * @param note
   * @param obj
   * @returns {Promise}
   */
  AlarmArchiveSpace.prototype.addNote = function (alarm, note, obj) {
    obj = objectify(obj);
    var cb = new Callback(obj.ok, obj.fail, obj.batch);
    baja.comm.addNote(baja.bson.encodeValue(alarm), note, cb);
    return cb.promise();
  };

  /**
   * Returns an object map of alarm class names -> alarm class display names
   * @param {Object} [obj] The object literal.
   * @param {Function} [obj.ok] (Deprecated: use Promise) Called once mapping
   * has been created.
   * @param {Function} [obj.fail] (Deprecated: use Promise) Called if the
   * mapping cannot be created.
   * @param {baja.comm.Batch} [obj.batch] if defined, any network calls will be
   * batched into this object
   * @returns {Promise.<Object>} Promise resolved with map object
   */
  AlarmArchiveSpace.prototype.getDisplayNamesMap = function (obj) {
    obj = objectify(obj);
    var cb = new Callback(obj.ok, obj.fail, obj.batch);
    baja.comm.getDisplayNamesMap(cb);
    return cb.promise();
  };

  /**
   * Returns an object map of alarm class names -> alarm class permissions
   *
   * @since Niagara 4.9
   *
   * @param {Array.<String>} alarmClasses The alarm class names for displayed alarms.
   * @param {Object} [obj] The object literal.
   * @param {Function} [obj.ok] (Deprecated: use Promise) Called once mapping
   * has been created.
   * @param {Function} [obj.fail] (Deprecated: use Promise) Called if the
   * mapping cannot be created.
   * @param {baja.comm.Batch} [obj.batch] if defined, any network calls will be
   * batched into this object
   * @returns {Promise.<Object>} Promise resolved with map object
   */
  AlarmArchiveSpace.prototype.getPermissionsMap = function (alarmClasses, obj) {
    obj = objectify(obj);
    var cb = new Callback(obj.ok, obj.fail, obj.batch);
    baja.comm.getPermissionsMap(baja.bson.encodeValue(alarmClasses.join(',')), cb);
    return cb.promise();
  };

  /**
   * Returns an array of the declared fields in an alarm record
   * @param {Object} [obj] The object literal.
   * @param {Function} [obj.ok] (Deprecated: use Promise) Called once mapping
   * has been created.
   * @param {Function} [obj.fail] (Deprecated: use Promise) Called if the
   * mapping cannot be created.
   * @param {baja.comm.Batch} [obj.batch] if defined, any network calls will be
   * batched into this object
   * @returns {Promise.<Array<String>>} Promise resolved with Array of Strings representing field names
   */
  AlarmArchiveSpace.prototype.getAlarmFields = function (obj) {
    obj = objectify(obj);
    var cb = new Callback(obj.ok, obj.fail, obj.batch);
    baja.comm.getAlarmFields(cb);
    return cb.promise();
  };

  /**
   * Acknowledge a group of alarms by id and/or source
   * @param {Object} [params]
   * @param {Array.<String>} [params.ids] Array of encoded baja:Uuid strings representing alarm ids
   * @param {Array.<String>} [params.srcs] Array of encoded baja.Ord strings representing sources
   * @param {Object} [obj] The object literal.
   * @param {Function} [obj.ok] (Deprecated: use Promise) Called once mapping
   * has been created.
   * @param {Function} [obj.fail] (Deprecated: use Promise) Called if the
   * mapping cannot be created.
   * @returns {Promise} Promise resolved when alarms have been acknowledged
   */
  AlarmArchiveSpace.prototype.ackAlarms = function (params, obj) {
    obj = objectify(obj);
    var cb = new Callback(obj.ok, obj.fail, obj.batch);
    baja.comm.ackAlarms(params, cb);
    return cb.promise();
  };

  /**
   * Add notes to a number of alarms by id and/or source.
   *
   * @param {Object} [params]
   * @param {Array.<String>} [params.ids] Array of encoded baja:Uuid strings representing alarm ids
   * @param {Array.<String>} [params.srcs] Array of encoded baja.Ord strings representing sources
   * @param {boolean} [params.a] value set as true to show that the view is invoked over the alarm archive space
   * @param {Object} [obj] The object literal.
   * @param {Function} [obj.ok] (Deprecated: use Promise) Called once the request has completed.
   * @param {Function} [obj.fail] (Deprecated: use Promise) Called if the request fails.
   * @returns {Promise} Promise resolved once the call is complete.
   */
  AlarmArchiveSpace.prototype.addNoteToAlarms = function (params, obj) {
    obj = objectify(obj);
    params.a = true;
    var cb = new Callback(obj.ok, obj.fail, obj.batch);
    baja.comm.addNoteToAlarms(params, cb);
    return cb.promise();
  };

  /**
   * Force alarms clear by id and/or source.
   *
   * @param {Object} [params]
   * @param {Array.<String>} [params.ids] Array of encoded baja:Uuid strings representing alarm ids
   * @param {Array.<String>} [params.srcs] Array of encoded baja.Ord strings representing sources
   * @param {Object} [obj] The object literal.
   * @param {Function} [obj.ok] (Deprecated: use Promise) Called once the request has completed.
   * @param {Function} [obj.fail] (Deprecated: use Promise) Called if the request fails.
   * @returns {Promise} Promise resolved once the call is complete.
   */
  AlarmArchiveSpace.prototype.forceClear = function (params, obj) {
    obj = objectify(obj);
    var cb = new Callback(obj.ok, obj.fail, obj.batch);
    baja.comm.forceClear(params, cb);
    return cb.promise();
  };

  /**
   * Query summary alarms for a single source.
   *
   * @param {Object} params
   * @param {Number} params.offset The query offset for the cursor. This is the index of where the cursor starts.
   * @param {Number} params.limit The limit to the number of records downloaded.
   * @param params.source The alarm source.
   * @param params.ord The ORD for the recipient to query alarms from.
   * @param [params.filterSet] An optional filter set.
   * @param [params.timeRange] An optional time range filter.
   * @param {String} [params.column] An optional column name to sort by. This can be an alarm record property name
   * or prefixed with 'alarmData.' if sorting via an alarm data facets column. If not specified, the alarm data
   * will be sorted by timestamp.
   * @param {Boolean} [params.sortDesc] An optional flag used to indicate whether to sort by descending or ascending order.
   * If not specified, the data will sort by descending order.
   * @param {Object} [obj] The object literal.
   * @param {Function} [obj.ok] (Deprecated: use Promise) Called once the request has completed.
   * @param {Function} [obj.fail] (Deprecated: use Promise) Called if the request fails.
   * @returns {Promise} Promise resolved once the call is complete.
   */
  AlarmArchiveSpace.prototype.getSingleSourceSummary = function (params, obj) {
    obj = objectify(obj);
    var cb = new Callback(obj.ok, obj.fail, obj.batch);
    params.source = String(params.source);
    params.ord = String(params.ord);
    if (params.filterSet) {
      params.filterSet = baja.bson.encodeValue(params.filterSet);
    }
    if (params.timeRange) {
      params.timeRange = baja.bson.encodeValue(params.timeRange);
    }
    baja.comm.getSingleSourceSummary(params, cb);
    return cb.promise();
  };

  /**
   * Returns notes for an alarm record.
   *
   * @param {Object} [params]
   * @param {Array.<String>} [params.uuid] Encoded baja:Uuid string representing an alarm id
   * @param {Object} [obj] The object literal.
   * @param {Function} [obj.ok] (Deprecated: use Promise) Called once mapping
   * has been created.
   * @param {Function} [obj.fail] (Deprecated: use Promise) Called if the
   * mapping cannot be created.
   * @param {baja.comm.Batch} [obj.batch] if defined, any network calls will be
   * batched into this object
   * @returns {Promise.<String>} Promise resolved with a notes string
   */
  AlarmArchiveSpace.prototype.getNotes = function (params, obj) {
    obj = objectify(obj);
    var cb = new Callback(obj.ok, obj.fail, obj.batch);
    baja.comm.getNotes(params, cb);
    return cb.promise();
  };
  return AlarmArchiveSpace;
});
