/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * @module baja/boxcs/FireTopicOp
 * @private
 */
define(["bajaScript/bson", "bajaScript/baja/boxcs/SyncOp"], function (baja, SyncOp) {
  "use strict";

  var subclass = baja.subclass,
    callSuper = baja.callSuper;
  var serverDecodeContext = baja.$serverDecodeContext;
  var bsonEncodeValue = baja.bson.encodeValue;

  /**
   * @class Fire Topic SyncOp.
   *
   * @name FireTopicOp
   * @extends SyncOp
   * @inner
   * @private
   *
   * @param {baja.Component} comp the Component the Topic is being fired from.
   * @param {baja.Slot} slot the Topic Slot.
   * @param {baja.Value|null} event the event to be fired (can be null).
   */
  function FireTopicOp(comp, slot, event) {
    callSuper(FireTopicOp, this, arguments);

    // Encode argument to a data structure
    var a = {
      nm: this.getId(),
      h: comp.getHandle(),
      n: slot.getName()
    };
    if (event !== null) {
      a.b = bsonEncodeValue(event); // Encode the value to BSON
    }
    this.$arg = a;
  }
  subclass(FireTopicOp, SyncOp);
  FireTopicOp.id = "t";
  function fireDecodedTopic(slot, comp, value, display) {
    comp.fire({
      slot: slot,
      value: value,
      cx: {
        commit: true,
        display: display,
        serverDecode: true
      }
    });
  }

  /**
   * Decode and commit the SyncOp.
   *
   * @private
   *
   * @param {baja.Component} comp the Component the Topic is being fired from.
   * @param {object} sp the syncOp data structure to be decoded.
   * @returns {Promise|*} if a topic is being fired, return a promise to be resolved after the fire
   * is complete.
   */
  FireTopicOp.decodeAndCommit = function (comp, sp) {
    if (!comp) {
      return;
    }

    // Decode and fire the Component event
    // TODO: Propagate amongst Knobs

    var slotName = sp.n,
      bson = sp.b;
    var slot = comp.getSlot(slotName);
    if (!slot) {
      return; // no topic to fire
    }
    if (!bson) {
      return fireDecodedTopic(slot, comp, null, ''); // topic fires with no event value
    }
    var display = bson.d;
    return baja.bson.decodeAsync(bson, serverDecodeContext).then(function (event) {
      return fireDecodedTopic(slot, comp, event, display);
    });
  };
  return FireTopicOp;
});
