/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.coll.Table}.
 * @module baja/coll/Table
 */
define(["bajaPromises", "bajaScript/bson", "bajaScript/baja/obj/Simple", "bajaScript/baja/coll/collUtil", "bajaScript/baja/coll/tableMixIn"], function (bajaPromises, baja, Simple, collUtil, tableMixIn) {
  "use strict";

  var subclass = baja.subclass,
    callSuper = baja.callSuper,
    isObjEmpty = collUtil.isObjEmpty;

  /**
   * Represents a `baja:ITable` in BajaScript.
   * 
   * Tables are usually returned as the result of resolving an ORD (i.e. a BQL query).
   *
   * @class
   * @alias baja.coll.Table
   * @mixes baja.coll.tableMixIn
   *
   * @see module:baja/coll/TableCursor
   * @see module:baja/coll/Table.TableColumn
   */
  var Table = function Table(tableData) {
    callSuper(Table, this, arguments);
    this.$tableData = tableData;
  };
  subclass(Table, Simple);
  tableMixIn(Table.prototype);

  /**
   * Default `Table` instance.
   * @type {baja.coll.Table}
   */
  Table.DEFAULT = new Table({});

  /**
   * Make a `Table`.
   * 
   * @private
   *
   * @param {Object} tableData
   * @returns {baja.coll.Table} the Table.
   */
  Table.make = function (tableData) {
    if (isObjEmpty(tableData)) {
      return baja.coll.Table.DEFAULT;
    }
    return new Table(tableData);
  };

  /**
   * Make a `Table`.
   * 
   * @private
   *
   * @param {Object} tableData
   * @returns {baja.coll.Table} the Table
   */
  Table.prototype.make = function (tableData) {
    return Table.make(tableData);
  };

  /**
   * Decode a `Table` from a `String`.
   * 
   * @private
   * 
   * @param {String} str
   * @returns {baja.coll.Table}
   */
  Table.prototype.decodeFromString = function (str) {
    return Table.make(JSON.parse(str));
  };

  /**
   * Encode the `Table` to a `String`.
   *
   * @private
   *
   * @returns {String}
   */
  Table.prototype.encodeToString = function () {
    return JSON.stringify(this.$tableData);
  };

  /**
   * Resolves to the string value of the given table's actual TypeSpec from the
   * table's config if context is provided.
   *
   * If no context provided, this will return the string value of the type
   * registered on this table class. This will be a `box:BoxTable` type.
   *
   * @param {Object} [cx]
   * @returns {Promise.<String>|String}
   */
  Table.prototype.toString = function (cx) {
    var typeString = this.getType().toString();
    if (!cx) {
      return typeString;
    }
    if (!this.toConfig) {
      return bajaPromises.resolve(typeString);
    }
    return this.toConfig().then(function (config) {
      return baja.importTypes({
        typeSpecs: [config.get('typeSpec')]
      });
    }).then(function (type) {
      return baja.lt(type).getDisplayName(cx);
    })["catch"](function () {
      return typeString;
    });
  };
  return Table;
});
