/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Raymond A. Richards
 */

/**
 * Defines {@link baja.LinkCheck}.
 * @module baja/comp/LinkCheck
 */
define([], function () {
  "use strict";

  /**
   * LinkCheck is the result of a call to {@link baja.Component#checkLink}.
   * It stores information about a link's validity, and if the LinkCheck
   * is invalid then it stores the reason.
   *
   * @class
   * @alias baja.LinkCheck
   * @param {boolean} valid
   * @param {String} [reason]
   */
  var LinkCheck = function LinkCheck(valid, reason) {
    this.valid = valid;
    this.invalidReason = reason;
  };

  /**
   * Indicates the validity of the LinkCheck
   * @returns {boolean} true if the LinkCheck is valid
   */
  LinkCheck.prototype.isValid = function () {
    return this.valid;
  };

  /**
   * Returns the reason a LinkCheck is invalid.
   * @returns {String} the reason the LinkCheck is invalid.
   */
  LinkCheck.prototype.getInvalidReason = function () {
    return this.invalidReason;
  };

  /**
   * Make a LinkCheck indicating a valid link.
   * @returns {baja.LinkCheck}
   */
  LinkCheck.makeValid = function () {
    return new LinkCheck(true, undefined);
  };

  /**
   * Make a LinkCheck indicating an invalid link.
   * This invalidReason should be localized for the Context passed to
   * {@link baja.Component#checkLink}.
   *
   * @param {String} invalidReason
   * @returns {baja.LinkCheck}
   */
  LinkCheck.makeInvalid = function (invalidReason) {
    return new LinkCheck(false, invalidReason);
  };

  /**
   * Decode a LinkCheck from JSON.
   *
   * @private
   * @param {object} resp JSON encoded LinkCheck
   * @param {boolean} [resp.v] true for a valid LinkCheck
   * @param {string} [resp.r] invalid reason
   * @returns {baja.LinkCheck}
   * @see BComponentSpaceSessionHandler.checkLink()
   */
  LinkCheck.fromJSON = function (resp) {
    return new LinkCheck(resp.v, resp.v ? undefined : resp.r);
  };

  /**
   * @typedef baja.LinkCheck~LinkCreateInfo
   * @property {baja.Component} source the source component to set on the `BLink`
   * @property {baja.Slot|string} sourceSlot the source slot to set on the `BLink`
   * @property {baja.Component} target the target component to set on the `BLink`
   * @property {baja.Slot|string} targetSlot the target slot to set on the `BLink`
   */

  return LinkCheck;
});
