/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Define {@link baja.Enum}.
 * @module baja/obj/Enum
 */
define(["bajaScript/sys", "bajaScript/baja/obj/Simple"], function (baja, Simple) {
  "use strict";

  var subclass = baja.subclass,
    callSuper = baja.callSuper;

  /**
   * Represents a `baja:Enum` in BajaScript.
   *
   * @class
   * @alias baja.Enum
   * @extends baja.Simple
   */
  var Enum = function Enum() {
    callSuper(Enum, this, arguments);
  };
  subclass(Enum, Simple);

  /**
   * Return the `Enum` (itself).
   *
   * @returns {baja.Enum}
   */
  Enum.prototype.getEnum = function () {
    return this;
  };

  /**
   * Return the ordinal of the `Enum` (itself).
   *
   * @returns {Number}
   */
  Enum.prototype.valueOf = function () {
    return this.getOrdinal();
  };

  /**
   * Return an `Enum` from a BIEnum.
   *
   * @param {baja.Complex|Object} enumVal - a `baja:StatusEnum`, a `baja.Complex` with a `baja:StatusEnum` `out` slot, or any object with a `getEnum` function defined.
   *
   * @returns {baja.Enum} the derived `Enum` value, or `baja.DynamicEnum.DEFAULT` if none present.
   *
   * @throws {Error} if a non-`baja.Value` was passed in
   */
  Enum.getEnumFromIEnum = function (enumVal) {
    var val = enumVal,
      type = enumVal.getType(),
      hasOut = false,
      out;
    if (type.isComplex()) {
      out = enumVal.get("out");
      if (out && out.getType().is("baja:StatusValue")) {
        val = out.getValue();
        hasOut = true;
      }
    }
    if (!hasOut) {
      if (type.is("baja:StatusValue")) {
        val = enumVal.getValue();
      } else if (typeof enumVal.getEnum === "function") {
        val = enumVal.getEnum();
      }
    }
    return val instanceof baja.Enum ? val : baja.DynamicEnum.DEFAULT;
  };

  /**
   * Return facets from an IEnum
   * 
   * @param {baja.Value} enumVal 
   * @returns {baja.Facets}
   */
  Enum.getFacetsFromIEnum = function (enumVal) {
    // First check if getBooleanFacets is available
    if (typeof enumVal.getEnumFacets === 'function') {
      return enumVal.getEnumFacets();
    }
    if (baja.hasType(enumVal, 'baja:StatusEnum')) {
      return baja.Facets.make({
        range: enumVal.getValue().getRange()
      });
    }

    // Next check the 'out' and 'facets' slots
    return baja.Facets.getFacetsFromObject(enumVal);
  };
  return Enum;
});
