/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.AlarmScheme}.
 * @module baja/ord/AlarmScheme
 */
define(["bajaScript/sys", "bajaScript/baja/ord/OrdQuery", "bajaScript/baja/ord/OrdScheme", "bajaScript/baja/ord/OrdTarget"], function (baja, OrdQuery, OrdScheme, OrdTarget) {
  "use strict";

  var subclass = baja.subclass,
    callSuper = baja.callSuper,
    alarmSpaceTypeSpec = "alarm:FileAlarmDatabase",
    alarmArchiveSpaceTypeSpec = "alarm:NullAlarmArchive",
    alarmSpacePromise,
    alarmArchiveSpacePromise;

  /**
   * Alarm ORD Scheme.
   * 
   * This scheme resolves to a alarm space.
   *
   * @class
   * @alias baja.AlarmScheme
   * @extends baja.OrdScheme
   * @private
   */
  var AlarmScheme = function AlarmScheme() {
    callSuper(AlarmScheme, this, arguments);
  };
  subclass(AlarmScheme, OrdScheme);

  /**
   * Default Alarm ORD Scheme instance.
   * @private
   * @type {baja.AlarmScheme}
   */
  AlarmScheme.DEFAULT = new AlarmScheme();
  function initAlarmSpace() {
    if (alarmSpacePromise) {
      return alarmSpacePromise;
    }
    var alarmSpace;
    baja.nav.localhost.alarm = alarmSpace = baja.nav.localhost.$addChildNode(baja.$(alarmSpaceTypeSpec));
    alarmSpacePromise = alarmSpace.init().then(function () {
      return alarmSpace;
    });
    return alarmSpacePromise;
  }
  function initAlarmArchiveSpace() {
    if (alarmArchiveSpacePromise) {
      return alarmArchiveSpacePromise;
    }
    var alarmArchiveSpace;
    return baja.importTypes([alarmArchiveSpaceTypeSpec]).then(function () {
      alarmArchiveSpaceTypeSpec = baja.$(alarmArchiveSpaceTypeSpec);
      baja.nav.localhost.alarm = baja.nav.localhost.$addChildNode(baja.$(alarmSpaceTypeSpec));
      baja.nav.localhost.alarm.archive = alarmArchiveSpace = baja.nav.localhost.alarm.$addChildNode(alarmArchiveSpaceTypeSpec);
      alarmArchiveSpacePromise = alarmArchiveSpace.init().then(function () {
        return alarmArchiveSpace;
      });
      return alarmArchiveSpacePromise;
    });
  }

  /**
   * Called when an ORD is resolved.
   *
   * @private
   *
   * @see baja.OrdScheme#resolve
   *
   * @param {module:baja/ord/OrdTarget} target  the current ORD Target.
   * @param {baja.OrdQuery} query  the ORD Query used in resolving the ORD.
   * @param {module:baja/ord/OrdQueryListCursor} cursor  the ORD Query List 
   * cursor used for helping to asynchronously resolve the ORD.
   * @param {Object} options  options used for resolving an ORD.
   */
  AlarmScheme.prototype.resolve = function (target, query, cursor, options) {
    var newTarget = new OrdTarget(target);
    function fail(err) {
      options.callback.fail(err);
    }
    if (!query.getBody()) {
      initAlarmSpace().then(function (alarmSpace) {
        newTarget.object = alarmSpace;
        cursor.resolveNext(newTarget, options);
      }, fail);
    } else if (query.getBody().indexOf('archive') !== -1) {
      initAlarmArchiveSpace().then(function (alarmArchiveSpace) {
        newTarget.object = alarmArchiveSpace;
        cursor.resolveNext(newTarget, options);
      }, fail);
    } else {
      // Resolve the ORD and force it to resolve on the Server.
      baja.Ord.make(options.ord).resolve({
        forceServerResolve: true
      }).then(function (target) {
        newTarget.object = target.object;
        cursor.resolveNext(newTarget, options);
      }, fail);
    }
  };

  /**
   * Return an ORD Query for the scheme.
   *
   * @returns {baja.OrdQuery}
   */
  AlarmScheme.prototype.parse = function (schemeName, body) {
    return new OrdQuery({
      scheme: this,
      schemeName: schemeName,
      body: body,
      isHost: false,
      isSession: false
    });
  };
  return AlarmScheme;
});
