/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.FileScheme}.
 * @module baja/ord/FileScheme
 */
define(["bajaScript/sys", "bajaScript/baja/ord/OrdQuery", "bajaScript/baja/ord/OrdScheme", "bajaScript/baja/ord/OrdTarget"], function (baja, OrdQuery, OrdScheme, OrdTarget) {
  "use strict";

  var subclass = baja.subclass,
    callSuper = baja.callSuper;

  /**
   * File ORD Scheme.
   * 
   * This scheme resolves to a history space.
   *
   * @class
   * @alias baja.FileScheme
   * @extends baja.OrdScheme
   * @private
   */
  var FileScheme = function FileScheme() {
    callSuper(FileScheme, this, arguments);
  };
  subclass(FileScheme, OrdScheme);

  /**
   * Default File ORD Scheme instance.
   * @private
   * @type {baja.FileScheme}
   */
  FileScheme.DEFAULT = new FileScheme();

  /**
   * Called when an ORD is resolved.
   *
   * @private
   *
   * @see baja.OrdScheme#resolve
   *
   * @param {module:baja/ord/OrdTarget} target  the current ORD Target.
   * @param {baja.OrdQuery} query  the ORD Query used in resolving the ORD.
   * @param {module:baja/ord/OrdQueryListCursor} cursor  the ORD Query List 
   * cursor used for helping to asynchronously resolve the ORD.
   * @param {Object} options  options used for resolving an ORD.
   */
  FileScheme.prototype.resolve = function (target, query, cursor, options) {
    var newTarget = new OrdTarget(target);
    function fail(err) {
      options.callback.fail(err);
    }

    // Resolve the ORD and force it to resolve on the Server.
    return baja.Ord.make(query.toString()).resolve({
      forceServerResolve: true
    }).then(function (target) {
      newTarget = new OrdTarget(target);
      newTarget.object = target.object;
      cursor.resolveNext(newTarget, options);
    }, fail);
  };

  /**
   * Return an ORD Query for the scheme.
   *
   * @returns {baja.OrdQuery}
   */
  FileScheme.prototype.parse = function (schemeName, body) {
    return new OrdQuery({
      scheme: this,
      schemeName: schemeName,
      body: body,
      isHost: false,
      isSession: false
    });
  };
  return FileScheme;
});
