/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * @module baja/tag/RelationTags
 */
define(["bajaScript/sys", "bajaScript/baja/tag/Id", "bajaScript/baja/tag/Tag", "bajaScript/baja/tag/ComponentTags"], function (baja, Id, Tag, ComponentTags) {
  "use strict";

  /**
   * A Tags implementation for a Relation object.
   *
   * @class
   * @alias module:baja/tag/RelationTags
   *
   * @param {baja.Relation} owner The Relation whose tags we're going to represent.
   */
  var RelationTags = function RelationTags(owner) {
    this.$owner = owner;
  };

  /**
   * Return the tag's value if it exists.
   * 
   * @param  {String|module:baja/tag/Id} id The Id
   * used for the search. This can be an Id or a qname for an Id.
   * @returns The tag's value or null if it doesn't exist.
   */
  RelationTags.prototype.get = function (id) {
    return this.$owner.getRelationTags().get(ComponentTags.idToSlotName(id), null);
  };

  /**
   * Return true if the specified tag is available.
   * 
   * @param  {String|module:baja/tag/Id} id The Id
   * used for the search. This can be an Id or a qname for an Id.
   * @returns Returns true if the specified tag is available.
   */
  RelationTags.prototype.contains = function (id) {
    return this.get(id) !== null;
  };

  /**
   * Returns a copy of the contained Tags array.
   * 
   * @returns {Array<module:baja/tag/Tag>} An array of Tag objects.
   */
  RelationTags.prototype.getAll = function () {
    var tags = [],
      facets = this.$owner.getRelationTags(),
      keys = facets.getKeys(),
      i,
      t;
    for (i = 0; i < keys.length; ++i) {
      t = new Tag(ComponentTags.slotNameToQname(keys[i]), facets.get(keys[i]));
      tags.push(t);
    }
    return tags;
  };

  /**
   * @returns {Boolean} Returns true if there are no Tag objects.
   */
  RelationTags.prototype.isEmpty = function () {
    return this.$owner.getRelationTags().getKeys().length === 0;
  };
  return RelationTags;
});
