/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/* eslint-env browser */

/**
 * API Status: **Private**
 * @module baja/env/ConnectionManager
 */
define(["bajaScript/comm", "bajaScript/env/HttpOnlyConnection", "bajaScript/env/WorkbenchConnection", "bajaScript/env/WebSocketConnection", "bajaScript/env/ReusedWebSocketConnection", "bajaScript/env/mux/muxUtils", "bajaPromises"], function (baja, HttpOnlyConnection, WorkbenchConnection, WebSocketConnection, ReusedWebSocketConnection, muxUtils, bajaPromises) {
  "use strict";

  var OPEN_CURLY = 123; //'}'.charCodeAt(0)
  var decodeFragment = muxUtils.decodeFragment;
  var fromBytes = muxUtils.fromBytes;

  /**
   * Opens and maintains a full-duplex data connection to the server, such as a
   * WebSocket.
   *
   * @class
   * @alias module:baja/env/ConnectionManager
   * @param {Function} processNext A function called to process the next frame.
   */
  var ConnectionManager = function ConnectionManager(processNext) {
    var that = this,
      sentFrames = that.$sentFrames = [];
    that.$connection = null;

    /** @implements module:baja/env/Connection~ConnectionEvents */
    that.$handlers = {
      close: function close(ev) {
        var i;
        if (!baja.isStopping() && !baja.isStopped()) {
          // If there are any outstanding messages then fail them.
          for (i = 0; i < sentFrames.length; ++i) {
            sentFrames[i].cb.fail(ev || "Unknown");
          }
          sentFrames.splice(0, sentFrames.length);
        }

        // Remove BajaScript from event mode.
        baja.comm.stopEventMode();

        // Call to process the next frame.
        processNext();
      },
      message: function message(ev) {
        var data = ev.data;
        return bajaPromises.resolve().then(function () {
          if (typeof data === 'string') {
            return data;
          } else {
            return blobToArrayBuffer(data).then(function (arrayBuffer) {
              return new Uint8Array(arrayBuffer);
            });
          }
        }).then(function (data) {
          that.$handleDataFromServer(data);
        })["catch"](function (e) {
          baja.error("Invalid BOX frame: " + data);
          baja.error("Cause: ");
          baja.error(e);
        });
      },
      open: function open() {
        // Put BajaScript into event mode.
        baja.comm.startEventMode();
      },
      error: function error(ev) {
        baja.error(ev);
      }
    };
  };
  function frameBodyValidForMyServerSession(messageBody) {
    return idValidForMyServerSession(messageBody.id);
  }

  /**
   * @param {module:baja/env/mux/BoxEnvelope~BoxFragment} fragment
   * @returns {boolean}
   */
  function fragmentValidForMyServerSession(fragment) {
    return idValidForMyServerSession(fragment.sessionId);
  }
  function idValidForMyServerSession(id) {
    return !id || id === baja.comm.getServerSessionId();
  }

  /**
   * @private
   * @param params
   * @returns {module:baja/env/Connection}
   */
  ConnectionManager.prototype.$makeConnection = function (params) {
    var handlers = this.$handlers,
      hasWebSockets = typeof WebSocket === 'function';

    // eslint-disable-next-line camelcase
    if (typeof niagara_wb_util_sendBox === 'function') {
      return new WorkbenchConnection(handlers, this);
    } else if (hasWebSockets && params) {
      return new ReusedWebSocketConnection(handlers, this, params);
    } else if (hasWebSockets) {
      return new WebSocketConnection(handlers, this);
    }
    return new HttpOnlyConnection();
  };

  /**
   * Starts the connection or short circuits if a connection is already available.
   *
   * @private
   * 
   * @param [params] Optional parameters used to create the underlying connection.
   * @returns {Promise} promise to be resolved when the connection is
   * authenticated and opened.
   */
  ConnectionManager.prototype.start = function (params) {
    var that = this;
    if (that.$connection) {
      return bajaPromises.resolve();
    }
    return that.$doAuthenticate(params).then(function () {
      return that.$doStart();
    });
  };

  /**
   * Instantiates the connection and uses it to authenticate with the server.
   *
   * @private
   * @param {object} [params] params to be used to instantiate the connection
   * @returns {Promise}
   */
  ConnectionManager.prototype.$doAuthenticate = function (params) {
    var connection = this.$connection = this.$makeConnection(params);
    return bajaPromises.resolve().then(function () {
      return connection.authenticate();
    });
  };

  /**
   * Starts up the authenticated connection. If the connection fails to start,
   * fall back to HTTP-only (no web sockets or interop) and try again (without
   * re-authenticating).
   * @private
   * @returns {Promise}
   */
  ConnectionManager.prototype.$doStart = function () {
    var that = this,
      connection = that.$connection;
    return bajaPromises.resolve().then(function () {
      return connection.start();
    })["catch"](function () {
      connection = that.$connection = new HttpOnlyConnection();
      return connection.start();
    });
  };

  /**
   * Send the data to the connection.
   *
   * @private
   *
   * @param {module:baja/env/Connection~FrameData} frameData The frame data to
   * send
   * @returns {Boolean} true if successfully sent. If false is returned then the
   * message can't be sent. It may be resent by another method (e.g. an HTTP
   * POST).
   */
  ConnectionManager.prototype.send = function (frameData) {
    var conn = this.$getOpenConnection();
    // If WebSockets aren't support or the frame
    // needs to be sent synchronously or the web socket isn't open yet
    // then send via HTTP.
    if (frameData.frame.$sync || !conn) {
      return false;
    }

    // Remember the BOX frame we've sent so we can reference it on the response.
    this.$sentFrames.push(frameData);

    // Send the JSON message through the WebSocket
    conn.send(frameData);

    // Return true so we can send it
    return true;
  };

  /**
   * Send the data as a beacon to the connection, if the connection implements
   * `sendBeacon`.
   *
   * @private
   *
   * @param {module:baja/env/Connection~FrameData} frameData The frame data to
   * send
   * @returns {Boolean} true if successfully sent. If false is returned then the
   * message can't be sent. It may be resent by another method (e.g. an HTTP
   * POST).
   */
  ConnectionManager.prototype.sendBeacon = function (frameData) {
    var frame = frameData.frame,
      syncFlag = frame.$sync,
      conn = this.$getOpenConnection();
    if (syncFlag !== 'beacon' || !conn || typeof conn.sendBeacon !== 'function') {
      return false;
    }
    conn.sendBeacon(frameData);
    return true;
  };

  /**
   * @returns {boolean} true if a secure connection is open
   * @since Niagara 4.14
   */
  ConnectionManager.prototype.isSecure = function () {
    var conn = this.$getOpenConnection();
    return !!(conn && conn.isSecure());
  };

  /**
   * @private
   * @returns {module:baja/env/Connection|null} the underlying connection, if it
   * is present and open.
   */
  ConnectionManager.prototype.$getOpenConnection = function () {
    var conn = this.$connection;
    return conn && conn.isOpen() && conn;
  };

  /**
   * Close any underlying connections.
   *
   * @private
   */
  ConnectionManager.prototype.close = function () {
    if (this.$connection) {
      this.$connection.close();
    }
  };

  /**
   * @private
   * @returns {module:baja/env/Connection~InternalConnection|null} the internal
   * connection or null if there isn't one.
   */
  ConnectionManager.prototype.getInternalConnection = function () {
    return this.$connection && this.$connection.getInternalConnection();
  };
  ConnectionManager.prototype.receiveFrame = function (boxFrame) {
    // Parse the response message into a BOX frame
    var i,
      sentFrames = this.$sentFrames,
      matchedFrame,
      frameData;

    // Bail if not a BOX frame
    if (!boxFrame || boxFrame.p !== "box") {
      throw new Error('BOX frame required');
    }
    if (typeof boxFrame.n === "number") {
      // Find a match for the incoming frame.
      for (i = 0; i < sentFrames.length; ++i) {
        frameData = sentFrames[i];
        if (frameData.frame.getSequenceNum() === boxFrame.n) {
          try {
            // Make OK callback
            frameData.cb.ok(boxFrame);
          } finally {
            matchedFrame = true;
            sentFrames.splice(i, 1);
            if (frameData.processed) {
              frameData.processed();
            }

            // eslint-disable-next-line no-unsafe-finally
            break;
          }
        }
      }
    }
    if (!matchedFrame) {
      var messages = boxFrame.m;
      // Process unsolicited BOX frames
      for (i = 0; i < messages.length; ++i) {
        var message = messages[i];
        // Handle unsolicited events
        if (message.t === "u") {
          var body = message.b;
          if (frameBodyValidForMyServerSession(body)) {
            baja.comm.handleEvents(body.d);
          }
        }
      }
    }
  };
  ConnectionManager.prototype.receiveFragment = function (fragment) {
    if (!baja.$mux.isEnabled()) {
      // this can happen if reusing a websocket and it receives a fragment from
      // the parent before baja.initFromSysProps completes
      return;
    }
    if (fragmentValidForMyServerSession(fragment)) {
      this.$connection.receiveFragment(fragment);
    }
  };
  ConnectionManager.prototype.$handleDataFromServer = function (data) {
    var first = data[0];
    if (typeof first === 'string') {
      first = first.charCodeAt(0);
    }
    var isJson = first === OPEN_CURLY;
    if (isJson) {
      var obj = JSON.parse(fromBytes(data));
      this.receiveFrame(obj);
    } else {
      var fragment = decodeFragment(data);
      return this.receiveFragment(fragment);
    }
  };

  /**
   * @param {Blob} blob
   * @returns {Promise<ArrayBuffer>}
   */
  function blobToArrayBuffer(blob) {
    if (typeof blob.arrayBuffer === 'function') {
      return blob.arrayBuffer();
    }
    var df = bajaPromises.deferred();
    var reader = new FileReader();
    reader.addEventListener('load', function (e) {
      if (reader.readyState === 2) {
        var error = reader.error;
        if (error) {
          df.reject(error);
        } else {
          df.resolve(reader.result);
        }
      }
    });
    reader.readAsArrayBuffer(blob);
    return df.promise();
  }
  return ConnectionManager;
});
