/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/* eslint-env browser */

/**
 * WebSocket Connection.
 *
 * API Status: **Private**
 * @module baja/env/WebSocketConnection
 */
define(["bajaScript/comm", "bajaScript/baja/comm/Callback", "bajaScript/env/Connection"], function (baja, Callback, Connection) {
  "use strict";

  // for artificial websocket latency performance testing
  // var minDelay = 0;
  // var webSocketLatency = 0;

  /**
   * A connection that uses a WebSocket as a connection.
   *
   * @class
   * @extends module:baja/env/Connection
   * @alias module:baja/env/WebSocketConnection
   * @param {module:baja/env/Connection~ConnectionEvents} handlers handlers
   * for connection events.
   * @param {module:baja/env/ConnectionManager} connectionManager the connection
   * manager responsible for processing BoxFrames via incoming websocket
   * messages
   */
  var _WebSocketConnection = function WebSocketConnection(handlers, connectionManager) {
    baja.callSuper(_WebSocketConnection, this, arguments);
    this.$secure = /^https/i.test(_WebSocketConnection.$getProtocol());
  };
  baja.subclass(_WebSocketConnection, Connection);

  /**
   * Make a new server session. This must be done using the default HTTP
   * mechanism before a web socket can be opened.
   * @return {Promise} promise to be resolved after the server session is
   * created
   */
  _WebSocketConnection.prototype.doAuthenticate = function () {
    var cb = new Callback();
    baja.comm.makeServerSession(cb);
    return cb.promise();
  };

  /**
   * Opens a new web socket to the station.
   */
  _WebSocketConnection.prototype.start = function () {
    var handlers = this.$handlers;
    var connection = this.$internalConnection = this.$createSocket();
    for (var name in handlers) {
      if (handlers.hasOwnProperty(name)) {
        connection.addEventListener(name, handlers[name], false);
      }
    }
  };

  /**
   * @returns {boolean} true if connected via HTTPS.
   */
  _WebSocketConnection.prototype.isSecure = function () {
    return this.$secure;
  };

  /**
   * @private
   * @returns {string}
   */
  _WebSocketConnection.$getProtocol = function () {
    return location.protocol;
  };

  /**
   * @private
   * @returns {WebSocket} web socket pointed at the BOX WebSocket acceptor
   */
  _WebSocketConnection.prototype.$createSocket = function () {
    var protocol = this.isSecure() ? 'wss' : 'ws',
      uri = protocol + '://' + location.host + '/wsbox';

    // TODO: We should list supported protocols in the WebSocket constructor?
    // https://developer.mozilla.org/en-US/docs/WebSockets/Writing_WebSocket_client_applications

    try {
      return new WebSocket(uri);
    } catch (e) {
      baja.error('WebSockets are available in this browser, but could not ' + 'construct a WebSocket connection to ' + uri);
      throw e;
    }
  };
  return _WebSocketConnection;
});
