/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/* eslint-env browser */

/**
 * Browser Environment Utility Library for BajaScript that uses jQuery.
 *
 * API Status: **Private**
 * @module baja/env/jQueryBrowserEnvUtil
 */
define("bajaBrowserEnvUtil", ["jquery", "bajaScript/sys", "bajaPromises"], function ($, baja, Promise) {
  "use strict";

  /**
   * Make an AJAX call.
   *
   * @name baja.browser.comm.sendHttp
   * @function
   * @private
   * @inner
   *
   * @param {String} type  the method to use (i.e. 'POST' or 'GET').
   * @param {String} uri  the URI used in the connection.
   * @param {Boolean} async  boolean flag indicating if this is an asynchronous or synchronous call.
   * @param {Object} callback  this object must have 'ok' and 'fail' functions.
   * @param {String} [data]  the data of the HTTP POST/PUT.
   * @param {Function} [processNext] if defined, called after the http call has finished.
   */
  function sendHttp(type, uri, async, callback, data, processNext) {
    $.ajax(uri, {
      type: type,
      async: async,
      contentType: "application/json",
      dataType: "json",
      data: data
    }).done(function (data) {
      callback.ok(data);
    }).fail(function (jqXHR) {
      if (!baja.isStopping()) {
        callback.fail(new baja.comm.HttpError(jqXHR));
      }
    }).always(function () {
      if (async && processNext) {
        processNext();
      }
    });
  }

  /**
   * Send the data to the target with no callback on completion. Godspeed
   * little data!
   *
   * This should be called instead of `sendHttp` during window unload handlers.
   *
   * @name baja.browser.comm.sendBeacon
   * @function
   * @private
   * @inner
   * @param {string} uri
   * @param {Object} callback  this object must have 'ok' and 'fail' functions.
   * @param {string} data
   * @param {Function} [processNext] if defined, called after the http call has finished.
   */
  function sendBeacon(uri, callback, data, processNext) {
    if (typeof navigator !== 'undefined' && typeof navigator.sendBeacon === 'function') {
      navigator.sendBeacon(uri, data);
      callback.ok();
      if (processNext) {
        processNext();
      }
    } else {
      sendHttp('POST', uri, false, callback, data);
    }
  }

  /**
   * Invoke the specified function once the DOM is ready.
   *
   * @name baja.browser.domReady
   * @function
   * @private
   * @inner
   *
   * @param {Function} func function to be called once the DOM is ready.
   */
  function domReady(func) {
    $(func);
  }

  /**
   * Add the promise specific behavior to the Callback class.
   *
   * This method will implement the <code>onOk</code>, <code>onFail</code>
   * and <code>promise</code> Callback methods.
   *
   * @name baja.browser.mixInPromises
   * @function
   * @private
   * @inner
   *
   * @param {Function} Callback the Callback constructor to mix the promises into.
   */
  function mixInPromises(Callback) {
    function initDeferred(cb) {
      // Lazily initialize the deferred promise object
      cb.$df = cb.$df || Promise.deferred();
      return cb.$df;
    }
    Callback.prototype.onOk = function (cx, args) {
      initDeferred(this).resolve(args[0]);
    };
    Callback.prototype.onFail = function (cx, args) {
      initDeferred(this).reject(args[0]);
    };
    Callback.prototype.promise = function () {
      return initDeferred(this).promise();
    };
  }
  return {
    sendBeacon: sendBeacon,
    sendHttp: sendHttp,
    domReady: domReady,
    mixInPromises: mixInPromises
  };
});
