/**
 * @private
 * @module nmodule/micros/rc/MicrosPointManager
 */
define(['baja!', 'underscore', 'nmodule/micros/rc/MicrosUtils', 'nmodule/webEditors/rc/wb/mgr/mgrUtils', 'nmodule/driver/rc/wb/mgr/PointMgr', 'nmodule/webEditors/rc/fe/baja/util/compUtils', 'nmodule/webEditors/rc/wb/mgr/MgrTypeInfo', 'nmodule/micros/rc/MicrosPointManagerModel', 'bajaux/commands/Command', 'nmodule/micros/rc/MicrosNewCommand', 'nmodule/micros/rc/DatabaseSyncCommand', 'nmodule/micros/rc/SyncSummaryCommand', 'nmodule/webEditors/rc/wb/mgr/commands/EditCommand', 'nmodule/webEditors/rc/wb/mgr/commands/NewCommand', 'nmodule/webEditors/rc/wb/mgr/commands/NewFolderCommand'], function (baja, _, MicrosUtils, mgrUtils, PointMgr, compUtils, MgrTypeInfo, MicrosPointManagerModel, Command, MicrosNewCommand, DatabaseSyncCommand, SyncSummaryCommand, EditCommand, NewCommand, NewFolderCommand) {
  'use strict';

  var TYPES_TO_LOAD = [MicrosUtils.typeSpec.BOOLEAN_POINT, MicrosUtils.typeSpec.MICROS_PROXY_EXT, MicrosUtils.typeSpec.MICROS_NETWORK, MicrosUtils.typeSpec.MICROS_POINT_FOLDER];

  /**
   * Constructor. This specifies a point folder type and a depth to use for a DepthSubscriber
   */
  var MicrosPointManager = function MicrosPointManager(params) {
    var that = this;
    PointMgr.call(that, _.extend({
      moduleName: 'micros',
      keyName: 'MicrosPointManager',
      folderType: MicrosUtils.typeSpec.MICROS_POINT_FOLDER,
      subscriptionDepth: 3
    }, params));
  };
  MicrosPointManager.prototype = Object.create(PointMgr.prototype);
  MicrosPointManager.prototype.constructor = MicrosPointManager;

  /**
   * Return a Promise that will resolve to the point model.
   */
  MicrosPointManager.prototype.makeModel = function (component) {
    return baja.importTypes({
      // make sure the types we need (for this module) are available locally
      typeSpecs: TYPES_TO_LOAD
    }).then(function /*types*/
    () {
      return MgrTypeInfo.make([baja.lt(MicrosUtils.typeSpec.BOOLEAN_POINT)]);
    }).then(function (newTypes) {
      return new MicrosPointManagerModel({
        component: component,
        newTypes: newTypes
      });
    });
  };

  /**
   * Commands for the Micros Point Manager
   *
   * @returns {Promise.<Array.<module:bajaux/commands/Command>>}
   */
  MicrosPointManager.prototype.makeCommands = function () {
    var that = this,
      i,
      newCommandIndex = -1,
      editCommandIndex = -1,
      newFolderCommandIndex = 0;
    return PointMgr.prototype.makeCommands.apply(that, arguments).then(function (superCmds) {
      var cmds = superCmds;

      // The superclass implementation of this method gives us New Folder, New & Edit,
      // We want New (a special Micros one), Edit, then New Folder.
      // On workbench there's also a Tagit button. It's not implemented in the UX manager,
      // but if it is at some point in the future, this implementation should pick it up automatically.

      // 1. Find the standard NewCommand and remove it.
      for (i = 0; i < cmds.length; i++) {
        if (cmds[i] instanceof NewCommand) {
          newCommandIndex = i;
          break;
        }
      }

      // remove the NewCommand if we found one
      if (newCommandIndex > -1) {
        cmds.splice(newCommandIndex, 1);
      }

      // 2. Add the MicrosNewCommand as the first command.
      cmds.splice(0, 0, new MicrosNewCommand(that));

      // 3. Add the DatabaseSync and SyncResult Commands after NewFolder
      // (or after the MicrosNewCommand if there isn't a NewFolder).
      for (i = 0; i < cmds.length; i++) {
        if (cmds[i] instanceof NewFolderCommand) {
          newFolderCommandIndex = i;
          break;
        }
      }
      cmds.splice(newFolderCommandIndex + 1, 0, new DatabaseSyncCommand(that), new SyncSummaryCommand(that));

      // 4. Find the EditCommand, remove it, then add it back after the MicrosNewCommand.
      for (i = 0; i < cmds.length; i++) {
        if (cmds[i] instanceof EditCommand) {
          editCommandIndex = i;
          break;
        }
      }

      // remove the EditCommand if we found one
      if (editCommandIndex > -1) {
        cmds.splice(editCommandIndex, 1);
      }
      cmds.splice(1, 0, new EditCommand(that));

      // A simpler implentation creating the commands list from scratch
      //    var that = this;
      //    var cmds = [new MicrosNewCommand(that),
      //                new NewFolderCommand(that),
      //                new DatabaseSyncCommand(that),
      //                new SyncSummaryCommand(that)
      //               ];

      return cmds;
    });
  };

  /**
   * Get the `NewCommand` instance from the command group or undefined if the command was not
   * added in the call to makeCommands.
   *
   * @private
   * @returns {module:bajaux/commands/Command}
   */
  MicrosPointManager.prototype.$getNewCommand = function () {
    return mgrUtils.findCommand(this, MicrosNewCommand);
  };
  MicrosPointManager.prototype.$getSyncSummaryCommand = function () {
    return mgrUtils.findCommand(this, SyncSummaryCommand);
  };

  //  /**
  //   * Returns the point ext for the current loaded component. As we may have a folder
  //   * loaded, this may be several levels up in the component tree.
  //   *
  //   * @returns {baja.Component}
  //   */
  //  MicrosPointManager.prototype.getPointExt = function () {
  //    var model = this.getModel()
  //        comp = model.getComponentSource().getContainer();
  //
  //    return compUtils.closest(comp, baja.lt('micros:MicrosPointDeviceExt'));
  //  };
  //
  //  /**
  //   * Return the device from the current component's point ext.
  //   *
  //   * @returns {baja.Component}
  //   */
  //  MicrosPointManager.prototype.getDevice = function () {
  //    var pointExt = this.getPointExt();
  //
  //    return this.getPointExt().getParent();
  //  };

  MicrosPointManager.prototype.getNetwork = function () {
    var model = this.getModel(),
      comp = model.getComponentSource().getContainer();
    return compUtils.closest(comp, baja.lt(MicrosUtils.typeSpec.MICROS_NETWORK));
  };
  return MicrosPointManager;
});
