/**
 * @file Basic Web App Container View.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*global niagara */

define(['baja!', 'lex!mobile', 'jquery', 'jquerymobile', 'Promise', 'underscore', 'mobile/util/mobile/mobile', 'mobile/util/mobile/commands', 'mobile/util/mobile/selectview', 'mobile/util/mobile/pages', 'mobile/util/mobile/views/ListView', 'mobile/util/mobile/views/PageView', 'mobile/util/mobile/views/PageViewManager', 'css!mobile/desktop/desktop'], function (baja, lexs, $, jqm, Promise, _, mobileUtil, commands, views, pages, ListView, PageView, PageViewManager) {

  // Use ECMAScript 5 Strict Mode
  "use strict";

  var AppsPageView,
      AppsListView,
      AppsPageViewManager,
      appsMgr,
      mobileLex = lexs[0];

  function getComponentName(component) {
    if (!(component instanceof baja.Complex)) {
      return undefined;
    }

    return component.getDisplayName() || component.getName();
  }

  ////////////////////////////////////////////////////////////////
  //Util
  ////////////////////////////////////////////////////////////////

  /**
   * A component view to display apps and their corresponding shiny buttons on
   * a user's desktop screen.
   * 
   * @class
   * @memberOf niagara.desktop
   * @extends niagara.util.mobile.ListView
   * @private
   */
  AppsListView = baja.subclass(function () {
    baja.callSuper(AppsListView, this, arguments);
  }, ListView);

  AppsListView.prototype.highlight = function () {};

  AppsListView.prototype.toDisplayName = function () {
    var complex = this.value(),
        name = getComponentName(complex);

    return Promise.resolve(name || mobileLex.get('propsheet.station'));
  };

  /**
   * After building a list item for an app, check to see if the view type 
   * spec for the app matches the current view  (`MobileDesktopView`). If not,
   * set  `data-ajax="false"` on the link to that app to trigger a full page
   * refresh to the app.
   * 
   * @name niagara.desktop.AppsListView#makeListItem
   * @function
   */
  AppsListView.prototype.makeListItem = function (complex, prop) {
    return baja.callSuper("makeListItem", AppsListView, this, arguments).then(function (li) {
      var appDisplayInfo = complex.get(prop),
          a = li.children('a');

      a.children('img').removeClass('ui-li-icon');
      a.jqmData('ord', baja.Ord.make({
        base: "station:",
        child: appDisplayInfo.get("appSlotPath")
      }));

      return li;
    });
  };

  /**
   * When subscribing a `NiagaraMobileWebApps` component, listen for the
   * `appsModified` topic to be fired. When it is fired, refresh the view to
   * display any added/removed apps.
   * 
   * @name niagara.desktop.AppsListView#makeSubscriber
   * @function
   */
  AppsListView.prototype.makeSubscriber = function () {
    var sub = baja.callSuper('makeSubscriber', AppsListView, this, arguments);

    sub.attach("topicFired", function (topic) {
      // If appModified event is fired then simply refresh the view
      if (topic.getName() === "appsModified") {
        appsMgr.refreshAll();
      }
    });

    return sub;
  };

  /**
   * A list item for an app should have the app display name and icon set.
   * App icons are always linkable and never expandable.
   * 
   * @name niagara.desktop.AppsListView#makeListItemParams
   * @function
   */
  AppsListView.prototype.makeListItemParams = function (component, prop) {
    var appDisplayInfo = component.get(prop);
    return Promise.resolve({
      title: '',
      display: appDisplayInfo.get("appDisplayName"),
      icon: appDisplayInfo.get("appDisplayIcon"),
      expandable: false,
      linkable: true
    });
  };

  /**
   * `AppsPageView` has a specialized implementation of `makeListView` that
   * performs a server side call to retrieve app display information
   * (`DesktopServerSideCallHandlers.getAppDisplayInfo()`). The output of this
   * server side call - not the  `NiagaraMobileWebApps` component itself - will
   * be passed into `makeListItem` to build up the display.
   * 
   * @name niagara.desktop.AppsListView#makeListView
   * @function
   */
  AppsListView.prototype.makeListView = function (component) {
    var ul = $('<ul />'),
        //no data-role="listview" - JQM should not enhance
    that = this;

    // Make a Server Side Call to get the information implemented by BIAppDisplay
    // (tried to do this with BQL but it became too complex so a Server Side Call is easiest)
    return component.serverSideCall({
      typeSpec: "mobile:DesktopServerSideCallHandler",
      methodName: "getAppDisplayInfo"
    }).then(function (result) {
      var makeListItems = _.map(result.getSlots().toArray(), function (slot) {
        return that.makeListItem(result, slot);
      });
      return Promise.all(makeListItems).then(function (lis) {
        return ul.html(lis);
      });
    });
  };

  /**
   * A component view to display apps and their corresponding shiny buttons on
   * a user's desktop screen.
   * 
   * @class
   * @memberOf niagara.desktop
   * @extends niagara.util.mobile.ComponentView
   * @private
   */
  AppsPageView = baja.subclass(function AppsPageView() {
    baja.callSuper(AppsPageView, this, arguments);
  }, PageView);

  AppsPageView.prototype.instantiateContentView = function () {
    return new AppsListView();
  };

  /**
   * Appends a view button to the page's header.
   * 
   * @name niagara.desktop.AppsPageView#createPage
   * @function
   */
  AppsPageView.prototype.createPage = function () {
    var page = baja.callSuper('createPage', AppsPageView, this, arguments),
        headerDiv = page.children(':jqmData(role=header)');

    headerDiv.append(commands.getCommandsButton());

    return page;
  };

  AppsPageViewManager = function AppsPageViewManager() {
    PageViewManager.call(this);
  };
  AppsPageViewManager.prototype = new PageViewManager();

  AppsPageViewManager.prototype.instantiateView = function instantiateView(component, pageData) {
    return Promise.resolve(new AppsPageView());
  };

  /**
   * Registers Pages handlers and instigates a page change to display the
   * current `AppView`. Will be called upon `baja.start`.
   * 
   * @name niagara.desktop.initializeUI
   * @function
   * @private
   */
  function initializeUI() {
    appsMgr = new AppsPageViewManager();
    appsMgr.registerPages();

    function showFirstPage() {
      $(document).on('onShowCommands', 'a.commandsButton', function () {
        var value = appsMgr.$selectedView.value();
        views.getViewsCommand().setOrd(value.getNavOrd());
      });

      mobileUtil.linkToOrdInternal(niagara.view.ord, {
        transition: "none",
        changeHash: false
      });
    }

    pages.register({
      pageshow: showFirstPage
    });
  }

  return initializeUI;
});
