/**
 * @file Base field editor from which all others inherit.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * @private
 * @module mobile/fieldeditors/BaseFieldEditor
 */
define(['baja!', 'bajaux/Widget', 'jquery', 'Promise', 'underscore', 'bajaux/events', 'bajaux/mixin/batchSaveMixin', 'nmodule/js/rc/asyncUtils/asyncUtils'], function (baja, Widget, $, Promise, _, events, batchSaveMixin, asyncUtils) {

  "use strict";

  var doRequire = asyncUtils.doRequire,
      COMMIT_READY = batchSaveMixin.COMMIT_READY,


  //copy-pasted from SimpleIntrospector.java
  dataTypeMap = {
    "baja.Boolean": 'b',
    "baja:Integer": 'i',
    "baja:Long": 'l',
    "baja:Float": 'f',
    "baja:Double": 'd',
    "baja:String": 's',
    "baja:DynamicEnum": 'e',
    "baja:EnumRange": 'E',
    "baja:AbsTime": 'a',
    "baja:RelTime": 'r',
    "baja:TimeZone": 'z',
    "baja:Unit": 'u'
  };

  var requireFe = _.once(function () {
    return doRequire('mobile/fieldeditors/fieldeditors');
  });

  /**
   * The base field editor. This editor will come with all functionality built
   * in except hooks for building the HTML, setting a value, and retrieving the
   * set value. This constructor should never be invoked directly - use
   * `niagara.fieldEditors.makeFor` method  instead.
   * 
   * @private
   * @class
   * @alias module:mobile/fieldeditors/BaseFieldEditor
   * @extends module:bajaux/Widget
   * @see niagara.fieldEditors.makeFor
   */
  var BaseFieldEditor = function BaseFieldEditor(value, container, slot, params) {
    Widget.call(this, 'mobile', 'BaseFieldEditor');

    if (value === undefined || value === null) {
      //for subclassing / prototype creation
      return this;
    }

    params = baja.objectify(params);

    if (value.getType().isComplex() && !value.getType().isComponent()) {
      value = value.newCopy();
    }

    this.$value = value;
    this.container = container || value.getParent && value.getParent();
    this.slot = slot || value.getName && value.getName();
    this.name = String(this.slot);

    this.params = params;
    this.facets = params.facets || baja.Facets.DEFAULT;
    this.label = params.label || container && container.getDisplayName && slot && container.getDisplayName(slot) || value.getName && value.getName() || slot && String(slot);

    this.parent = params.parent;

    this.$enabled = !params.readonly;

    this.$kids = [];

    this.postCreate();
    batchSaveMixin(this);
  };
  BaseFieldEditor.prototype = Object.create(Widget.prototype);
  BaseFieldEditor.prototype.constructor = BaseFieldEditor;

  BaseFieldEditor.prototype.setModified = function (modified, params) {
    if (this.isLoading()) {
      return;
    }

    Widget.prototype.setModified.apply(this, [modified]);

    params = baja.objectify(params);

    var that = this,
        parent = that.parent,
        dom = that.jq();

    if (modified) {
      if (parent) {
        parent.setModified(true, $.extend({ silent: true }, params));
      }

      if (dom && !params.silent) {
        dom.trigger('modified.view', [that, params]);
      }
    }
  };

  /**
   * A field editor subclass may, optionally, define some extra initialization
   * code here to set up the state of the editor after the constructor
   * completes. The default behavior is to do nothing.
   */
  BaseFieldEditor.prototype.postCreate = function () {};

  /**
   * Sets a field editor's enabled status. Default behavior is simply to set
   * the `disabled` attribute on any `input` or `select` elements contained in
   * the field editor's DOM.
   *
   * @param {Boolean} enabled
   */
  BaseFieldEditor.prototype.doEnabled = function doEnabled(enabled) {
    var element = this.jq(),
        inputs = element && element.find('select,input,textarea');

    if (inputs) {
      if (enabled) {
        inputs.removeAttr('disabled');
      } else {
        inputs.attr('disabled', 'disabled');
      }
    }
  };

  function isCheckable(element) {
    return element.is('input[type="checkbox"]') || element.is('input[type="radio"]');
  }

  BaseFieldEditor.$detectChangeEvent = function () {
    var element = $(this),
        editorDiv = element.closest('div.editor'),
        editor = editorDiv.data('editor'),
        oldValue = element.data('oldValue'),
        value;

    if (isCheckable(element)) {
      //for JQM checkboxes "val()" always seems to return "on"
      value = element.is(':checked');
    } else {
      value = element.val();
    }

    if (!editor.isLoading() && oldValue !== undefined && oldValue !== value) {
      editor.setModified(true, {
        oldValue: oldValue,
        newValue: value,
        element: element
      });
    }

    element.data('oldValue', value);
  };

  /**
   * A quick way of performing `initialize` and `load` in one step. The value
   * passed to `load` will be the same passed into the constructor as
   * `params.value`.
   * 
   * @param {jQuery} targetElement the div in which the editor should build ids
   * html
   * @returns {Promise}
   */
  BaseFieldEditor.prototype.buildAndLoad = function buildAndLoad(targetElement) {
    var that = this;

    return that.initialize(targetElement).then(function () {
      return that.load(that.value());
    });
  };

  /**
   * Adds CSS classes to a div that correspond to the given Type and all of
   * its supertypes. The classes will be of the form `module-Type`, e.g.
   * `baja-Component`. The colon will be replaced with a  hyphen as a matter of
   * CSS compatibility.
   * 
   * @memberOf niagara.fieldEditors
   * @private
   * @param {JQuery} div the div on which to add CSS classes
   * @param {Type} type
   */
  function addSuperTypeClasses(div, type) {
    baja.iterate(type, function (type) {
      div.addClass(String(type.getTypeSpec()).replace(':', '-'));
    }, function (type) {
      return type.getSuperType();
    });
  }

  /**
   * Adds CSS classes reflecting the whole supertype chain - so that your CSS
   * can target divs with a CSS class of `baja-StatusValue`, for instance.
   *
   * @param {baja.Value} value
   * @returns {Promise}
   */
  BaseFieldEditor.prototype.load = function (value) {
    var that = this;

    return Widget.prototype.load.apply(that, arguments).then(function (value) {
      var dom = that.jq();

      if (value !== null && value !== undefined) {
        addSuperTypeClasses(dom, value.getType());
      }

      dom.find('input,select,textarea').each(function () {
        var $this = $(this);
        if (isCheckable($this)) {
          $this.data('oldValue', $this.is(':checked'));
        } else {
          $this.data('oldValue', $this.val());
        }
      });

      return value;
    });
  };

  /**
   * A BaseFieldEditor will wrap its content in a div with `class="editor"`.
   * It will also give this div `data('editor', this)`.
   * @param {JQuery} element
   */
  BaseFieldEditor.prototype.initialize = function (element) {
    element = $('<div class="editor ui-field-contain"/>').data('editor', this).appendTo(element);

    element.on('change keyup', 'input,select,textarea', BaseFieldEditor.$detectChangeEvent);

    return Widget.prototype.initialize.apply(this, [element]);
  };

  /**
   * @memberOf niagara.fieldEditors
   * @private
   */
  function toFakeBajaObject(stringEncoded, type) {
    //getDataTypeSymbol is required in order to encode a simple as part of a
    //Facets object

    var obj = baja.$(type).decodeFromString(stringEncoded),
        symbol = dataTypeMap[type.toString()];

    obj.getDataTypeSymbol = function () {
      return symbol;
    };

    return obj;
  }
  /**
   * Pulls any user-entered data from the editor div's input/select elements,
   * and assembles them into an actual value object.
   * 
   * @returns {Promise} the value constructed from user-entered data 
   */
  BaseFieldEditor.prototype.read = function read() {
    var that = this,
        value = that.value(),
        myType = baja.hasType(value) && value.getType();

    return Widget.prototype.read.apply(that, arguments).then(function (readValue) {
      if (myType && myType.is('baja:Simple') && !myType.is('baja:String') && readValue.getType().is('baja:String')) {
        return toFakeBajaObject(readValue, myType);
      } else {
        return readValue;
      }
    });
  };

  /**
   * Check to see if the field editor is violating the edit-by-ref
   * requirement for component editors.
   * @ignore
   */
  function componentMismatch(readValue, currentValue) {
    var readType = readValue.getType && readValue.getType();

    if (!readType || !readType.isComponent()) {
      return false;
    }

    if (readValue.editedSlots) {
      return false; //this is a save data component used for validation
    }

    return readValue !== currentValue;
  }

  /**
   * Sequential async workflow that retrieves save data from a field editor,
   * performs the save (performing a server-side update, if we are editing a
   * mounted component), triggers `saved.fieldeditor` on the field editor's DOM
   * element, and resets the modified flag to false.
   *
   * @param {Object} params
   * @returns {Promise}
   */
  BaseFieldEditor.prototype.save = function save(params) {
    var that = this,
        readValue,
        readType;

    return that.validate().then(function (r) {
      readValue = r;
      readType = r.getType && r.getType();

      if (componentMismatch(readValue, that.value())) {
        throw new Error("Component editors must support edit-by-ref semantics");
      }

      return that.doSave(readValue, params);
    }).then(function () {
      var container = that.container,
          slot = that.slot,
          batch = params && params.batch,
          progressCallback = params && params.progressCallback;

      if (readType && !readType.isComponent() && container && slot) {
        //since we're not editing a Component, the batch could not have been
        //used in doSave - so it's good to be used here instead
        var prom = container.set({
          slot: slot,
          value: readValue,
          batch: batch
        });

        if (progressCallback) {
          progressCallback(COMMIT_READY);
        }

        return prom;
      } else {
        //no need to save - components get edited in-place
        return readValue;
      }
    }).then(function () {
      that.trigger(events.SAVE_EVENT);
      that.setModified(false);
      that.$value = readValue;
    });
  };

  /**
   * The default `doSave` behavior used by a field editor when a custom
   * implementation of `doSave` is not passed in to `defineEditor`. It handles a
   * specific type of data passed out of `doRead` - if you choose to allow the
   * default `doSave` behavior, your implementation of `doRead` must return data
   * in this format. This format will be described below.
   *
   * 
   * - If editing a `baja:Simple`, simply return a new instance of the edited
   *   type. For instance, if editing a `baja:Double`, return
   *   `baja.Double.make(1.23)`.
   * - If editing a `baja:Complex`, you have two choices.
   *   
   * -- Return an object literal where keys correspond to slot names and
   *    values are either `baja:Simples` (if the slot type is `baja:Simple`) or
   *    further object literals following the same format (if the slot type is
   *    `baja:Complex`). For example: `{ title: 'aString', subComponent: {
   *    title: 'aString' } }`
   * -- Or, pass your edited object into
   *    `niagara.fieldEditors.toSaveDataComponent`, set its slots directly, and
   *    return that. The advantage of this approach is that it functions as a
   *    sort of "preview" of the final saved object - `validate()` can examine
   *    it as if it were the actual edited object, already saved. You can also
   *    pass the output of `doRead()` into any server side calls that expect a
   *    `BValue` - useful for performing server-side validation.
   *
   * @param {baja.Complex|Object} [readValue] the save data output from `doRead`
   * @param {Object} params
   */
  BaseFieldEditor.prototype.doSave = function doSave(readValue, params) {
    var valueSets = [],
        valueToSave = this.value();

    //saving a BSimple
    if (valueToSave.getType().isSimple()) {
      return readValue;
    }

    //saving a BComplex
    if (readValue instanceof baja.Component && !readValue.editedSlots) {
      throw new Error("provided raw instance of Component (type " + readValue.getType() + ") to defaultDoSave - does your field " + "editor use niagara.fieldEditors.toSaveDataComponent()?");
    }

    var batch = new baja.comm.Batch();

    //if using toSaveDataComponent, readValue will have an editedSlots
    //property containing only those slots that were actually changed -
    //otherwise, just treat readValue as an object literal
    baja.iterate(readValue.editedSlots || readValue, function (slotValue, slotName) {
      valueSets.push(valueToSave.set({
        slot: slotName,
        value: slotValue.newCopy(true),
        batch: batch
      }));
    });

    batch.commit();

    return Promise.all(valueSets).then(function () {
      return valueToSave;
    });
  };

  /**
   * Instantiates a sub-editor belonging to this editor. The child editor
   * will remain aware of who its parent editor is, so whenever the child
   * is set to modified, its parent will be as well. It will automatically
   * be instantiated inside the same DOM element as the parent, unless a
   * different element is specified.
   * 
   * @param {Object} params a parameters object (same as the parameter to
   * makeFor)
   * @returns {Promise}
   */
  BaseFieldEditor.prototype.makeChildFor = function (params) {
    //TODO: once we have a concrete Type, store kids as dynamic slots
    var that = this,
        kids = that.$kids;

    params.element = params.element || that.jq();

    if (params.readonly === undefined) {
      params.readonly = !that.isEnabled();
    }

    return requireFe().then(function (fe) {
      return fe.makeFor(params);
    }).then(function (editor) {
      kids.push(editor);
      editor.parent = that;
      return editor;
    });
  };

  return BaseFieldEditor;
});
