/**
 * @file Composite field editors.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'jquery', 'Promise', 'underscore', 'mobile/util/slot', 'mobile/fieldeditors/BaseFieldEditor', 'mobile/fieldeditors/fieldeditors'], function (baja, $, Promise, _, slotUtil, BaseFieldEditor, fe) {

  "use strict";

  /**
   * Functions for creating composite field editors.
   * @private
   * @exports mobile/fieldeditors/fieldeditors.composite
   */

  var exports = {};

  function allSlots(editedObject) {
    var obj = {};
    editedObject.getSlots().properties().each(function (slot) {
      if (slotUtil.isEditable(slot)) {
        obj[String(slot)] = 'default';
      }
    });
    return obj;
  }

  /**
   * Initializes all sub-editors.
   *
   * @see module:mobile/fieldeditors/BaseFieldEditor#doInitialize
   */
  function compositeInitialize(slots) {
    return function (element, params) {
      var that = this,
          value = that.value(),
          makes = [],
          inits = [];

      //run all the makeFor calls first, sequentially, then run all
      //the initialize calls.
      //this is so any subeditors that might want to arrange themselves into
      //JQM elements, etc., can all be enhanced at once (otherwise things like
      //button groupings will fail).

      baja.iterate(slots || allSlots(value), function (feObj, slotName) {
        var subElement = $('<div class="subEditor"/>').appendTo(element),
            slot = value.getSlot(slotName),
            displayName = value.getDisplayName(slot);

        makes.push(that.makeChildFor({
          key: feObj.key,
          value: feObj.defaultValue || value.get(slotName),
          slot: slot,
          container: value,
          facets: that.facets,
          autoInitialize: false
        }).then(function (editor) {
          var lc = fe.toLabeledEditorContainer(displayName, subElement);
          that.subEditorMap[slot] = editor;
          inits.push(editor.initialize(lc, params));
          return editor;
        }));
      });

      return Promise.all(makes).then(function () {
        return Promise.all(inits);
      });
    };
  }

  /**
   * Instantiates, build and loads field editor instances for all desired 
   * slots on our component.
   *
   * @see module:mobile/fieldeditors/BaseFieldEditor#load
   * 
   * @param {Object} slots a mapping of slot names to field editor keys,
   * defining what types of field editors we want for our properties
   * @returns {Function} a setter function that takes in a baja.Complex, pulls
   * out that Complex's properties using the given slot names, and calls
   * load on each of our sub-field editors.
   */
  function compositeLoad(slots) {
    return function (value, params) {
      var that = this,
          loads = [],
          subEditorMap = that.subEditorMap;

      if (value === null || value === undefined) {
        return Promise.resolve(value);
      }

      baja.iterate(slots || allSlots(value), function (feObj, slotName) {
        var editor = subEditorMap[slotName],
            subValue = feObj.defaultValue || value.get(slotName);
        loads.push(editor.load(subValue, params));
      });

      return Promise.all(loads);
    };
  }

  /**
   * Retrieves the values from all of our individual field editors and
   * assembles them into an instance of our desired object type.
   * 
   * @private
   * @memberOf niagara.fieldEditors.composite
   * @see module:mobile/fieldeditors/BaseFieldEditor#read
   * 
   * @param {Object} slots a mapping of slot names to field editor keys,
   * defining what types of field editors we want for our properties
   * @returns {Function} a retriever function that will assemble our
   * individual field editor values into our desired composite type.
   */
  function compositeRead(slots) {
    return function () {
      var subEditorMap = this.subEditorMap,
          promises = [],
          value = this.value(),
          saveData = fe.toSaveDataComponent(value);

      baja.iterate(slots || allSlots(value), function (feObj, slotName) {
        var editor = subEditorMap[slotName];

        promises.push(editor.read().then(function (readValue) {
          return saveData.set({
            slot: slotName,
            value: readValue
          });
        }));
      });

      return Promise.all(promises).then(function () {
        return saveData;
      });
    };
  }

  /**
   * Sets the enabled status of all sub-field editors.
   *
   * @see module:mobile/fieldeditors/BaseFieldEditor#doEnabled
   * 
   * @param {Object} obj a mapping of slot names to field editor keys,
   * defining what types of field editors we want for our properties
   * @returns {Function} a function that will set enabled status on all
   * our sub-field editors.
   */
  function compositeDoEnabled(obj) {
    return function (enabled) {
      var subEditorMap = this.subEditorMap,
          value = this.value();
      baja.iterate(obj || allSlots(value), function (value, slotName) {
        var editor = subEditorMap[slotName];
        editor.setEnabled(enabled);
      });
    };
  }

  /**
   * Validates a composite field editor by sequentially calling the validate()
   * method of all of its sub-field editors. Returns a validate step to be
   * passed directly to `validators()` during the construction of the composite
   * field editor.
   *
   * @see module:mobile/fieldeditors/BaseFieldEditor#validate
   * 
   * @param {Object} obj a mapping of slot names to field editor keys,
   * defining what types of field editors we want for our properties
   * @returns {Function} a function that will set validate all of our
   * sub-field editors.
   */
  function compositeValidateStep(obj) {
    return function () {
      var promises = [],
          subEditorMap = this.subEditorMap,
          value = this.value();

      baja.iterate(obj || allSlots(value), function (value, slotName) {
        var editor = subEditorMap[slotName];
        promises.push(editor.validate());
      });

      return Promise.all(promises);
    };
  }

  /**
   * Creates a composite field editor. This field editor essentially breaks
   * our edited object down into individual properties, creates a field
   * editor for each one (choosing automatically depending on each property's
   * type), and assembles them into one big field editor.
   * 
   * For instance, the composite editor for control:EnumOverride is
   * simply defined by two slot names: duration and value. Since these two
   * properties of an EnumOverride are Numeric and Enum respectively, the
   * field editor created will contain two editor divs, one for the numeric
   * and one for the enum, and upon saving will assemble those editor's values
   * into an EnumOverride object.
   *
   * @param {Array} names the slot names of the properties we wish to show
   * for this editor
   * 
   * @param {Function} [FieldEditor] the field editor constructor to subclass
   * - if omitted, defaults to BaseFieldEditor
   * 
   * @returns {Function} a field editor constructor
   */
  exports.makeComposite = function (names, FieldEditor, params) {
    names = names || [];
    baja.strictArg(names, Array);
    var slots;

    if (names.length) {
      slots = {};
      baja.iterate(names, function (val) {
        if (typeof val === 'string') {
          slots[val] = {
            key: 'default'
          };
        } else {
          slots[val.slot] = {
            key: val.key || 'default',
            defaultValue: val.defaultValue
          };
        }
      });
    }

    return fe.defineEditor(FieldEditor || BaseFieldEditor, {
      doInitialize: compositeInitialize(slots),
      doLoad: compositeLoad(slots),
      doRead: compositeRead(slots),
      doEnabled: compositeDoEnabled(slots),
      validate: compositeValidateStep(slots),
      postCreate: function postCreate() {
        this.subEditorMap = {};
        if (params && params.validate) {
          this.validators().add(params.validate);
        }
      }
    });
  };

  /**
   * Returns a field editor to simply perform default editing of all
   * property slots on a BComplex.
   *
   * @returns {Function} a new field editor constructor
   */
  exports.allSlots = function () {
    var AllSlots = exports.makeComposite(),
        oldDoInitialize = AllSlots.prototype.doInitialize;

    AllSlots.prototype.doInitialize = function (element) {
      element.addClass('allSlots');
      oldDoInitialize.call(this, element);
    };

    return AllSlots;
  };

  return exports;
});
