/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * @private
 * @module mobile/fieldeditors/mobile/NumericEditor
 */
define(['baja!', 'jquery', 'Promise', 'mobile/fieldeditors/MobileFieldEditor', 'mobile/fieldeditors/mobile/StringEditor', 'mobile/fieldeditors/mobile/feUtils'], function (baja, $, Promise, MobileFieldEditor, StringEditor, feUtils) {

  'use strict';

  var getFacets = feUtils.getFacets,
      minMaxCheck = feUtils.minMaxCheck;

  /**
   * @private
   * @class
   * @alias module:mobile/fieldeditors/mobile/NumericEditor
   * @extends module:mobile/fieldeditors/MobileFieldEditor
   */

  var NumericEditor = function NumericEditor() {
    var _this = this;

    MobileFieldEditor.apply(this, arguments);
    this.validators().add(function (value) {
      var facets = getFacets(_this);
      if (facets) {
        return minMaxCheck(_this, facets.get("min"), facets.get("max"), value);
      }
    });
  };
  NumericEditor.prototype = Object.create(MobileFieldEditor.prototype);
  NumericEditor.prototype.constructor = NumericEditor;

  NumericEditor.prototype.doInitialize = function (dom) {
    var wrapper = $('<div class="numericWrapper"/>'),
        label = $('<label class="unitsDisplay"/>').appendTo(wrapper),
        span = $('<span class="inputWrapper"/>').appendTo(wrapper),
        input = $('<input type="text" data-theme="c"/>').attr('name', this.name).appendTo(span);

    return getDisplayUnits(this).then(function (units) {
      if (units) {
        //if the unit is prefixed, we want to move the symbol over to the left
        if (units.getIsPrefix()) {
          label.css('order', '0');
          input.css('text-align', 'left');
        }

        label.text(units.getSymbol());
        label.show();
      } else {
        label.hide();
      }

      wrapper.appendTo(dom);
    });
  };

  NumericEditor.prototype.doLoad = function (value) {
    var that = this,
        number = value.valueOf(),
        facets = getFacets(that),
        precision = facets && facets.get('precision');

    return getDisplayUnits(that).then(function (displayUnits) {
      if (displayUnits) {
        var realUnits = getRealUnits(that);
        number = realUnits.convertTo(displayUnits, number);
      }

      if (isNumber(number) && isNumber(precision)) {
        number = number.toFixed(Math.min(precision, 20));
      }

      return StringEditor.prototype.doLoad.call(that, number);
    });
  };

  NumericEditor.prototype.doRead = function () {
    var oldType = this.value().getType(),
        val = StringEditor.prototype.doRead.call(this);

    return readNumber(this).then(function (number) {
      if (oldType.is('baja:Integer')) {
        switch (val.toLowerCase()) {
          case "min":
            return baja.Integer.MIN_VALUE;
          case "max":
            return baja.Integer.MAX_VALUE;
          default:
            if (isNaN(number)) {
              throw new Error(val + " is not a valid Integer value");
            } else {
              return baja.Integer.make(number);
            }
        }
      } else if (oldType.is('baja:Long')) {
        switch (val.toLowerCase()) {
          case "min":
            return baja.Long.MIN_VALUE;
          case "max":
            return baja.Long.MAX_VALUE;
          default:
            if (isNaN(number)) {
              throw new Error(val + " is not a valid Long value");
            } else {
              return baja.Long.make(number);
            }
        }
      } else if (oldType.is('baja:Float')) {
        switch (val.toLowerCase()) {
          case "nan":
            return baja.Float.NAN;
          case "-inf":
            return baja.Float.NEGATIVE_INFINITY;
          case "+inf":
            return baja.Float.POSITIVE_INFINITY;
          default:
            if (isNaN(number)) {
              throw new Error(val + " is not a valid Float value");
            } else {
              return baja.Float.make(number);
            }
        }
      } else if (oldType.is('baja:Double')) {
        switch (val.toLowerCase()) {
          case "nan":
            return baja.Double.NaN;
          case "-inf":
            return baja.Double.NEGATIVE_INFINITY;
          case "+inf":
            return baja.Double.POSITIVE_INFINITY;
          default:
            if (isNaN(number)) {
              throw new Error(val + " is not a valid Double value");
            } else {
              return baja.Double.make(number);
            }
        }
      } else {
        if (isNaN(number)) {
          throw new Error(val + " is not a valid numeric value");
        } else {
          return number;
        }
      }
    });
  };

  function getDisplayUnits(ed) {
    var facets = getFacets(ed),
        units = facets.get('units'),
        unitConversion = facets.get('unitConversion');

    if (!units || units.equals(baja.Unit.NULL)) {
      return Promise.resolve(null);
    }

    if (!unitConversion) {
      return Promise.resolve(units);
    }

    return baja.UnitDatabase.get().then(function (db) {
      var ordinal = getConversionOrdinal(unitConversion),
          tag = baja.$('baja:UnitConversion', ordinal).getTag();

      return db.convertUnit(tag, units);
    });
  }

  function getConversionOrdinal(unitConversion) {
    if (baja.hasType(unitConversion, 'baja:Enum')) {
      return unitConversion.getOrdinal();
    }

    if (baja.hasType(unitConversion, 'baja:Number')) {
      return unitConversion.valueOf();
    }

    return null;
  }

  function getRealUnits(ed) {
    var units = getFacets(ed).get('units');

    return units && !units.equals(baja.Unit.NULL) ? units : null;
  }

  function readNumber(ed) {
    var number = Number(StringEditor.prototype.doRead.call(ed));

    return getDisplayUnits(ed).then(function (displayUnits) {
      if (displayUnits && !isNaN(number)) {
        number = displayUnits.convertTo(getRealUnits(ed), number);
      }
      return number;
    });
  }

  function isNumber(num) {
    return baja.hasType(num, 'baja:Number');
  }

  return NumericEditor;
});
