/**
 * @file The Data Table view extends the View framework, making use of the doLoad
 * method to instantiate a Data Table. The data table is created using one
 * or more ORDs to gather tabular data.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Jason Spangler
 */

define(['baja!history:HistoryExt', 'baja!', 'bajaux/Widget', 'jquery', 'jquerymobile', 'Promise', 'mobile/util/mobile/mobile', 'mobile/history/table/DataTableModel', 'css!mobile/history/table/DataTable'], function (types, baja, Widget, $, jqm, Promise, mobileUtil, DataTableModel) {

  // Use ECMAScript 5 Strict Mode
  "use strict";

  //local vars

  var columns,
      inAnimation = false,
      theme = 'b',
      //niagara.view.profile.theme,
  colDisplayState = {},
      escapeHtml = mobileUtil.escapeHtml,


  //constants
  SCROLL_DIV_CLASS = 'NavTableMain',
      DISPLAY_DIV_CLASS = 'tableDisplay',
      TABLE_CONTAINER_CLASS = 'tableContainer',
      tblColHeader = '<th class="ui-navbar ui-bar-' + theme + ' fade {id}" data-sort="{name}">' + '<span class="ui-btn ui-btn-up-' + theme + ' {icon}" style="border:0;margin:0;" >' + '<span class="ui-btn-inner" aria-hidden="true">' + '<span class="ui-btn-text">{display}</span>' + '</span>' + '</span>' + '</th>',
      tblIdxDisplay = '<tr><th colspan="{colspan}" class="ui-navbar ui-bar-' + theme + ' fade" >' + '<span class="recordIdxText">{offset} - {limit}</span></th></tr>';

  /**
   * Default fail callback handler.
   * 
   * @memberOf niagara.mobile.table
   */
  function defFail(err) {
    baja.error(err);

    // Hide any loading title      
    mobileUtil.hidePageLoadingMsg();

    // Load error page     
    $("<a href='#error' data-rel='dialog' data-transition='flip' />").click();
  }

  /**
   * This method returns a column display name for the view column header 
   * with appropriate facet information.
   * 
   * @memberOf niagara.mobile.table
   * 
   * @param {Object} column the Column object being parsed for the display name.
   * 
   * @returns {String} display name for the column along with unit symbol if present. 
   */
  function makeColumnHeaderDisplayValue(column) {
    var facets = column.getFacets(),
        units = facets ? facets.get('units') : undefined,
        display = column.getDisplayName(),
        unitSymbol;

    if (units) {
      units = units.encodeToString().split(";");
      unitSymbol = units[1];
      if (unitSymbol !== "" && unitSymbol !== "null") {
        display += " " + unitSymbol;
      }
    }

    return escapeHtml(display);
  }

  /**
   * Shows/hides columns in the table depending on what columns are currently
   * selected.
   * 
   * @memberOf niagara.mobile.table
   * @private
   * @param {jQuery} tableDiv
   */
  function showHideColumns(tableDiv) {
    //check to see what columns are currently visible
    var nm;
    for (nm in colDisplayState) {
      if (colDisplayState[nm] !== true) {
        $('.' + nm, tableDiv).hide();
      }
    }
  }

  /**
   * Advances the column sort state to the next state (none -> asc -> desc),
   * and rebuilds the table to reflect the new sort order. Will be called
   * whenever a sortable table has a column header clicked.
   * 
   * @memberOf niagara.mobile.table
   * @inner
   * @returns {Promise}
   */
  function doSort(view, model, colName) {
    var colSortState = model.getSortOrder(),
        sortedCol = colSortState.getColumn();

    //mark our column sort state
    if (sortedCol === undefined || sortedCol !== colName) {
      model.setSortOrder(colName, 'ASC');
    }
    //if our column name is already sorted, we either move to DESCENDING
    //sort state or we turn off sorting.
    else if (sortedCol === colName) {
        if (colSortState.isOrderAsc()) {
          model.setSortOrder(colName, 'DESC');
        } else {
          model.setSortOrder(colName, 'NONE');
        }
      }

    return view.makeTable(model);
  }

  /**
   * Creates a click handler for a column header cell. Will perform the sort,
   * build the table, and swap it out for the currently displayed table.
   * 
   * @memberOf niagara.mobile.table
   * @private
   * @inner
   * @returns Function
   */
  function makeClickHandler(view, model) {
    return function () {
      var $this = $(this),
          colName = $this.attr('data-sort');

      doSort(view, model, colName).then(function (tableDisplay) {
        var curDisplay = $this.parents('.' + DISPLAY_DIV_CLASS),
            targetDiv = curDisplay.parent();

        curDisplay.fadeOut(100, function () {
          curDisplay.remove();
          tableDisplay.appendTo(targetDiv).hide().fadeIn(100, function () {
            view.onpageswap(tableDisplay);
          });
        });
      }).catch(defFail);
    };
  }

  /**
   * This method is used to create the table header for a data table. Each
   * column is updated with a click event that performs sorting for the
   * given data column represented.
   * 
   * @param {niagara.mobile.table.DataTableView} view
   * @param {niagara.mobile.table.DataTableModel} model
   * 
   * @returns {jQuery} table row of table header elements.
   */
  function makeTableHeader(view, model) {
    var i,
        id,
        header,
        icon = '',
        cols = view.columns,
        colHeaders = $('<tr />'),
        colSortState = model.getSortOrder(),
        sortedCol = colSortState.getColumn(),
        clickHandler;

    if (model.isSortable()) {
      clickHandler = makeClickHandler(view, model);
    }

    //iterate through our columns and build <th> elements, including
    //icons representing current sort state as necessary.
    for (i = 0; i < cols.length; ++i) {
      id = escapeHtml(cols[i].getDisplayName().replace(/\s/g, '-'));

      //check sort state for column value
      if (sortedCol !== undefined && sortedCol === cols[i].getName()) {
        if (colSortState.isOrderAsc()) {
          icon = 'ui-btn-icon-right ui-icon-arrow-u';
        } else if (colSortState.isOrderDesc()) {
          icon = 'ui-btn-icon-right ui-icon-arrow-d';
        }
      } else {
        icon = '';
      }

      //generate our column header HTML
      header = $(tblColHeader.patternReplace({
        id: id,
        name: cols[i].getName(),
        icon: icon,
        display: makeColumnHeaderDisplayValue(cols[i])
      }));

      header.appendTo(colHeaders);

      if (clickHandler) {
        header.click(clickHandler);
      }

      //update our current column display state
      if (colDisplayState[id] === undefined) {
        colDisplayState[id] = true;
      }
    }

    return colHeaders;
  }

  /**
   * Enable or disable the next table button in the table view based on flag value.
   * 
   * @param {Boolean} flag - enable or disable button.
   * @param {jQuery} footer footer div
   */
  function enableNextBtn(flag, footer) {
    if (!footer) {
      return;
    }

    var nxtBtn = footer.find('.nextTable');
    nxtBtn.toggleClass('ui-disabled', !flag);
  }

  /**
   * Enable or disable the prev table button in the table view based on flag value.
   * 
   * @param {Boolean} flag - enable or disable button.
   * @param {jQuery} footer footer div
   */
  function enablePrevBtn(flag, footer) {
    if (!footer) {
      return;
    }

    var prevBtn = footer.find('.prevTable');
    prevBtn.toggleClass('ui-disabled', !flag);
  }

  //Data Table View


  /**
   * Constructor for our history table.
   * 
   * @memberOf niagara.mobile.table
   *
   * @class 
   * @param {Object} [options]
   *          - the Object Literal for the method's arguments.
   *          
   * @param {Number} [options.limit] 
   *          - Row limit for the display table. When the data is traversed
   *            for the ORD record set, the data table display will only 
   *            contain rows up to the specified limit.
   */
  var DataTableView = function DataTableView(options) {
    Widget.apply(this, arguments);
    options = baja.objectify(options, 'limit');

    this.mainDiv = $('<div class="' + SCROLL_DIV_CLASS + '" />');
    this.limit = options.limit || 20;
    this.model = undefined;
  };
  DataTableView.prototype = Object.create(Widget.prototype);
  DataTableView.prototype.constructor = DataTableView;

  /**
   * This method builds a table from the ORD assigned to this view.
   * 
   * @memberOf niagara.mobile.table
   * 
   * @param {Object} model
   *        - @see dataTable.model.js
   *        
   * @returns {Promise} promise to be resolved with the created table display
   * element
   */
  DataTableView.prototype.makeTable = function makeTable(model) {

    var offset = model.getOffset(),
        limit = model.getLimit(),
        table = $('<table border="1" />'),
        cols,
        rowCount = 0,
        that = this;

    return new Promise(function (resolve, reject) {
      model.get({
        cursor: {
          offset: offset,
          limit: limit,
          each: function each() {
            var row = $('<tr class="dataRow"/>'),
                i,
                id,
                displayValue;

            for (i = 0; i < columns.length; ++i) {
              id = escapeHtml(columns[i].getDisplayName().replace(/\s/g, "-"));
              displayValue = escapeHtml(this.getDisplay(columns[i])) || '&nbsp;';
              row.append($('<td class="' + id + '" >' + displayValue + '</td>'));
            }

            rowCount++;
            table.append(row);
          },
          after: function after() {
            var idxDisplay,
                tblDisplayDiv = $('<div class="' + DISPLAY_DIV_CLASS + '" />'),
                tableContainerDiv = $('<div class="' + TABLE_CONTAINER_CLASS + '"/>');

            //create our display and return
            idxDisplay = $(tblIdxDisplay.patternReplace({
              colspan: String(columns.length),
              offset: rowCount ? String(offset + 1) : '0',
              limit: String(offset + rowCount)
            }));

            table.prepend(idxDisplay);
            tableContainerDiv.append(table);

            tblDisplayDiv.append(tableContainerDiv);

            showHideColumns(tblDisplayDiv);

            //return the table display so it can be appended to a DOM element 
            resolve(tblDisplayDiv);
          }
        }
      }).then(function (history) {
        var colHeaders, valueCol;

        cols = history.getColumns();
        valueCol = history.getCol('value');
        if (valueCol) {
          that.$recordType = valueCol.getType();
        }

        that.columns = columns = cols;
        colHeaders = makeTableHeader(that, model);
        table.append(colHeaders);
      }).catch(reject);
    });
  };

  /**
   * The doLoad method is used to load the target box collection
   * (synonymous to a BITable) into our data view. 
   * 
   * @memberOf niagara.mobile.table.DataTableView#
   * 
   * @param {baja.Ord} value
   *          - The ORD to resolve to the data table contents.
   *          
   * @returns {Promise} promise to be resolved  once the data table is rendered
   * to HTML
   */
  DataTableView.prototype.doLoad = function (value) {

    var that = this;

    function loadModel(ord) {
      that.ord = baja.strictArg(ord, 'baja:Ord');
      that.model = new DataTableModel({ ord: that.ord, limit: that.limit });

      //append our view DOM to the target DOM element
      that.jq().append(that.mainDiv);

      //create a table from our model and use to build our view display
      return that.makeTable(that.model).then(function (tblDisplay) {
        var rowCount = tblDisplay.find('tr.dataRow').length;

        //check that our table is non-empty before appending to view
        that.mainDiv.html(tblDisplay);
        tblDisplay.show();
        //TODO: this is not working in a mobile px page because the main view is display: none while rendering
        that.mainDiv.height(tblDisplay.height());

        //if we are displaying less than the model limit, disable next
        enableNextBtn(rowCount >= that.model.getLimit(), that.$footer);

        // if our offset is 0, disable previous button
        enablePrevBtn(that.model.getOffset() > 0, that.$footer);

        that.onpageswap(tblDisplay);
      });
    }

    return baja.Ord.make(value).get().then(function (obj) {
      var type = obj.getType();

      if (type.is("history:HistoryExt")) {
        var historyConfig = obj.getHistoryConfig();
        return historyConfig.lease().then(function () {
          var historyId = historyConfig.getId();
          return loadModel(baja.Ord.make("history:" + historyId));
        });
      } else {
        return loadModel(value);
      }
    });
  };

  DataTableView.prototype.bindFooter = function (footer) {
    var that = this;

    that.$footer = footer;

    mobileUtil.preventNavbarHighlight(footer);

    footer.on('click', 'a', function () {
      var a = $(this);

      if (a.hasClass('ui-disabled')) {
        return false; //IE
      }
      that.swapTables(a.hasClass('nextTable')).catch(defFail);
    });
  };

  /**
   * This method detaches the current table and displays a new table in the
   * table pane. The contents of the table are determined by the records 
   * offset and whether the user selects to view the next set of records or
   * the previous set of records.
   * 
   * @memberOf niagara.mobile.table.DataTableView#
   * 
   * @param isNext
   */
  DataTableView.prototype.swapTables = function (isNext) {
    //check if we're currently mid-swap
    if (inAnimation) {
      return Promise.resolve();
    }

    inAnimation = true;

    var tblSlidingOff,
        tblPosOffSet,
        that = this,
        footer = that.$footer,
        pane = that.mainDiv.find("." + DISPLAY_DIV_CLASS),
        curPos = pane.position(),
        screenWidth = that.mainDiv.parents(':jqmData(role=content)').width();

    //calculate set index based on the current index and whether
    //we are getting previous records or first records
    if (isNext) {
      this.model.next();

      //animate tables sliding in from off-screen right
      tblPosOffSet = screenWidth + 15;
      tblSlidingOff = { left: -tblPosOffSet };
    } else {
      this.model.previous();

      //animate tables sliding in from off-screen left
      tblPosOffSet = -pane.width() - 15;
      tblSlidingOff = { left: curPos.left + screenWidth };
    }

    that.mainDiv.css('overflow', 'hidden');

    //construct our new table and animate into our view
    return that.makeTable(this.model).then(function (tblDisplay) {
      var count = tblDisplay.find('.' + TABLE_CONTAINER_CLASS + ' table tr.dataRow').length;

      //check if we have any data rows in our table
      if (count > 1) {
        tblDisplay.css('left', tblPosOffSet).css('right', tblPosOffSet + tblDisplay.width());

        that.mainDiv.append(tblDisplay);

        //slide off our old table
        pane.animate(tblSlidingOff, 'fast', function () {
          pane.remove();
        });

        tblDisplay.animate({ left: curPos.left }, 'fast', function () {
          inAnimation = false;
          that.mainDiv.css('overflow', 'visible');
          that.onpageswap(tblDisplay);
        });

        //if we have less rows that our limit allows, disable next
        if (count < that.model.getLimit()) {
          enableNextBtn(false, footer);
        } else {
          enableNextBtn(true, footer);
        }

        //if we are currently at our first record (0), diable prev
        if (that.model.getOffset() === 0) {
          enablePrevBtn(false, footer);
        }
      }
      //no data rows means we've reached the end of our data collection
      else {
          enableNextBtn(false, footer);
          that.model.previous();
          inAnimation = false;
        }

      //if our offset if greater than 0, enable our prev button
      if (that.model.getOffset() > 0) {
        enablePrevBtn(true, footer);
      }
    });
  };

  DataTableView.prototype.toggleColumnDisplay = function (columnName) {
    var selector = '#checkbox-' + columnName,
        checked = $(selector).is(':checked'),
        column = $('.' + columnName);

    column.toggle(checked);
    colDisplayState[columnName] = checked;
  };

  /**
   * Will be called each time a new table is loaded (initial load, column 
   * toggle, column sort, or page change).
   * 
   * @memberOf niagara.mobile.table.DataTableView#
   * 
   * @param {jQuery} dom the new DOM
   */
  DataTableView.prototype.onpageswap = function (dom) {};

  /**
   * Declare our namespace in the views space.
   */
  return DataTableView;
});
