/**
 * @file
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*global baja, $, niagara, describe, it, expect */

(function createScheduleTestUtils() {
  "use strict";

  var util = niagara.util,
      mock = niagara.test.mock.mock,
      fe = niagara.fieldEditors;

  function mockSchedule(typeSpec, params) {
    var days = 'sunday monday tuesday wednesday thursday friday saturday'.split(' '),
        sched = mock(typeSpec, params).addProp('schedule', mock('schedule:CompositeSchedule').addProp('week', mock('schedule:WeekSchedule'))).addProp('defaultOutput', mock('baja:StatusString').addProp('value', 'stringValue'));

    util.iterate(days, function (dayName) {
      var dailySchedule = mock('schedule:DailySchedule').addProp('day', mock('schedule:DaySchedule')).addProp('days', mock('schedule:AbstractSchedule'));
      sched.get('schedule').get('week').addProp(dayName, dailySchedule);
    });

    return sched;
  }

  function makeDay() {
    var schedule, dailySchedule, day;
    schedule = mockSchedule('schedule:BooleanSchedule');
    dailySchedule = mock('schedule:DailySchedule').addProp('day', mock('schedule:DaySchedule')).addProp('name', 'mockDailySchedule');
    day = fe.getFieldEditor(dailySchedule.getDay(), { key: 'day-ui-readonly' });
    util.iterate(Array.prototype.slice.call(arguments), function (obj) {
      day.addTimeSchedule(baja.Time.make(obj.start), baja.Time.make(obj.finish), obj.value);
    });

    return day;
  }

  util.api("niagara.util.test.schedule", {
    mockSchedule: mockSchedule,
    makeDay: makeDay
  });
})();

(function testScheduleUtil() {
  "use strict";

  var util = niagara.util,
      schedUtil = util.schedule,
      hourMillis = util.time.MILLIS_IN_HOUR;

  describe("schedule util", function testScheduleUtil() {

    describe("millisToPixels", function millisToPixels() {
      it("converts milliseconds to pixel height", function makesPixels() {
        function expectPixels(div, millis, pixels) {
          expect(schedUtil.millisToPixels(div, millis)).toEqual(pixels);
        }

        var div = $('<div/>').append($('<div class="blocksDisplay"/>').height(100));
        expect(div.children('.blocksDisplay').outerHeight()).toEqual(100);

        expectPixels(div, 0, 0);
        expectPixels(div, hourMillis * 6, 25);
        expectPixels(div, hourMillis * 12, 50);
        expectPixels(div, hourMillis * 24, 100);
      });
    });

    describe("pixelsToMillis", function pixelsToMillis() {
      it("converts pixel height to milliseconds", function makesMillis() {
        function expectMillis(div, pixels, millis) {
          expect(schedUtil.pixelsToMillis(div, pixels)).toEqual(millis);
        }

        var div = $('<div/>').append($('<div class="blocksDisplay"/>').height(100));
        expect(div.children('.blocksDisplay').outerHeight()).toEqual(100);

        expectMillis(div, 0, 0);
        expectMillis(div, 25, hourMillis * 6);
        expectMillis(div, 50, hourMillis * 12);
        expectMillis(div, 100, hourMillis * 24);
      });
    });

    describe("percentageOfDay", function percentageOfDay() {
      it("gives day percent from millis", function givesPercentFromMillis() {
        expect(schedUtil.percentageOfDay(0)).toEqual(0);
        expect(schedUtil.percentageOfDay(hourMillis * 6)).toEqual(0.25);
        expect(schedUtil.percentageOfDay(hourMillis * 12)).toEqual(0.5);
        expect(schedUtil.percentageOfDay(hourMillis * 24)).toEqual(1);
      });
    });
  });
})();
