var _slicedToArray = function () { function sliceIterator(arr, i) { var _arr = []; var _n = true; var _d = false; var _e = undefined; try { for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) { _arr.push(_s.value); if (i && _arr.length === i) break; } } catch (err) { _d = true; _e = err; } finally { try { if (!_n && _i["return"]) _i["return"](); } finally { if (_d) throw _e; } } return _arr; } return function (arr, i) { if (Array.isArray(arr)) { return arr; } else if (Symbol.iterator in Object(arr)) { return sliceIterator(arr, i); } else { throw new TypeError("Invalid attempt to destructure non-iterable instance"); } }; }();

/**
 * @file Datebox utilities for the schedule app.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'lex!mobile', 'jquery', 'jqmDatebox', 'Promise', 'underscore', 'mobile/util/time', 'mobile/fieldeditors/mobile/feUtils'], function (baja, lexs, $, jqmDatebox, Promise, _, timeUtil, feUtils) {

  "use strict";

  var getBajaTimeFormat = feUtils.getBajaTimeFormat,
      getDateboxDefaultOptions = feUtils.getDateboxDefaultOptions,
      DAY = timeUtil.MILLIS_IN_DAY,
      MINUTE = timeUtil.MILLIS_IN_MINUTE,
      _lexs = _slicedToArray(lexs, 1),
      mobileLex = _lexs[0],
      IS_DATEBOX_SELECTOR = ':jqmData(role=datebox)';

  /**
   * @private
   * @exports mobile/schedule/util.schedule.datebox
   */


  var exports = {};

  /**
   * We've attempted to move back the start time of a schedule block to a time
   * that conflicts with another block - find the earliest we can possibly
   * set it to without conflicts.
   *  
   * @param {module:mobile/schedule/DayEditor} dayEditor the day editor
   * for the day we're moving blocks around in
   * @returns {number} the earliest time we can set the start to without
   * conflicts
   */
  function earliestStartMillis(dayEditor) {
    var scheduleBlock = dayEditor.getSelectedBlock(),
        earliestms = 0,
        startms = scheduleBlock.start.getTimeOfDayMillis();

    dayEditor.getBlocks().forEach(function (myBlock) {
      if (myBlock !== scheduleBlock) {
        var finishms = myBlock.finish.getTimeOfDayMillis();

        if (finishms > 0 && finishms <= startms && (!earliestms || earliestms < finishms)) {
          earliestms = finishms;
        }
      }
    });

    return earliestms;
  }

  /**
   * We've attempted to move up the finish time of a schedule block to a time
   * that conflicts with another block - find the latest we can possibly
   * set it to without conflicts.
   *
   * @param {module:mobile/schedule/DayEditor} dayEditor the day editor
   * for the day we're moving blocks around in
   * @returns {number} the latest time we can set the finish to without
   * conflicts
   */
  function latestFinishMillis(dayEditor) {
    var scheduleBlock = dayEditor.getSelectedBlock(),
        latestms = DAY,
        finishms = scheduleBlock.finish.getTimeOfDayMillis();

    dayEditor.getBlocks().forEach(function (myBlock) {
      if (myBlock !== scheduleBlock) {
        var startms = myBlock.start.getTimeOfDayMillis();

        if (startms >= finishms && (!latestms || latestms > startms)) {
          latestms = startms;
        }
      }
    });

    return latestms % DAY;
  }

  /**
   * Check to see if our time schedule is valid - that is, the start time
   * is before the end time (or the end time is 12 midnight), and both
   * the start and finish times do not conflict with any other schedule
   * blocks in our day.
   * 
   * @param {Object} params an object literal containing parameters
   * @param {module:mobile/schedule/DayEditor} params.dayEditor the day
   * editor we're moving blocks around in
   * @param {baja.Time} params.startTime the current (pre-edit) start time
   * @param {baja.Time} params.finishTime the current (pre-edit) finish time
   * @param {String} params.slot the slot we wish to edit ('start' or 'finish')
   * @param {Number} params.mschange the change in ms we wish to make
   * @returns {Promise} promise to be resolved if change is valid, or rejected
   * with an appropriate message from the lexicon
   */
  exports.validateTimeChange = function (params) {
    var dayEditor = params.dayEditor,
        startTime = params.startTime,
        finishTime = params.finishTime,
        slot = params.slot,
        mschange = params.mschange,
        scheduleBlock = dayEditor.getSelectedBlock(),
        startms = startTime.getTimeOfDayMillis(),
        finishms = finishTime.getTimeOfDayMillis(),
        errorLexKey;

    if (slot === 'start') {
      startms += mschange;
      //start can't cross midnight boundary
      if (startms < 0 || startms >= DAY) {
        errorLexKey = 'schedule.message.crossesMidnight';
      }
    } else {
      finishms = finishms || DAY;
      finishms += mschange;
      //finish can't cross midnight boundary
      if (finishms <= 0 || finishms > DAY) {
        errorLexKey = 'schedule.message.crossesMidnight';
      }
    }

    //can't overlap with another block
    if (!dayEditor.canMoveBlockTo(scheduleBlock, startms, finishms)) {
      errorLexKey = 'schedule.message.blockOverlap';
    }

    //can't have finish before start
    if ((finishms || DAY) <= startms) {
      errorLexKey = 'schedule.message.finishBeforeStart';
    }

    if (errorLexKey) {
      return Promise.join(startTime.toTimeString({ textPattern: getBajaTimeFormat() || 'HH:mm' }), finishTime.toTimeString({ textPattern: getBajaTimeFormat() || 'HH:mm' })).spread(function (startStr, finishStr) {
        throw mobileLex.get({
          key: errorLexKey,
          args: [startStr, finishStr]
        });
      });
    } else {
      return Promise.resolve();
    }
  };

  /**
   * We've gotten an invalid time from a TimeSchedule editor - this
   * function returns a corrected, valid time. Prevents such errors as
   * finish time before start time, crossing over midnight, and overlapping
   * with another schedule block.
   *
   * @param {module:mobile/schedule/DayEditor} dayEditor the field
   * editor for the currently edited day
   * @param {Number} startms the current (pre-edit) start time for the 
   * TimeSchedule (in ms past midnight)
   * @param {Number} finishms the current (pre-edit) finish time for the 
   * TimeSchedule (in ms past midnight)
   * @param {String} slot the slot we are editing on the TimeSchedule
   * ('start' or 'finish')
   * @param {Number} mschange the change, in milliseconds, we are making
   * to the edited slot
   */
  function getCorrectedTime(dayEditor, startms, finishms, slot, mschange) {
    var direction = mschange > 0 ? 'down' : 'up',
        newstartms = startms + (slot === 'start' ? mschange : 0),
        newfinishms = finishms + (slot === 'finish' ? mschange : 0);

    newstartms = (newstartms + DAY) % DAY;
    newfinishms = (newfinishms + DAY) % DAY;

    switch (slot + ' ' + direction) {
      case 'start down':
        return (newfinishms || DAY) - MINUTE;
      case 'start up':
        return earliestStartMillis(dayEditor);
      case 'finish down':
        if (finishms + mschange > DAY) {
          return 0;
        } else {
          return latestFinishMillis(dayEditor);
        }
        break;
      case 'finish up':
        return newstartms + MINUTE;
      default:
        throw "slot must be one of 'start' or 'finish'";
    }
  }

  /**
   * Checks to ensure that the given time schedule (output from the field
   * editor on the editBlock page) is valid - if not, reset the start/finish
   * time on the datebox editor to the earliest/latest time that is valid. Call
   * this method each time the `datebox` event is triggered from your field
   * editor.
   * 
   * @memberOf niagara.util.schedule.datebox
   * @param {JQuery} input the `datebox` input element we're
   * changing (either the start or finish slot)
   * @param {module:mobile/schedule/DayEditor} dayEditor the 
   * field editor for the currently edited day
   * @param {module:mobile/fieldeditors/BaseFieldEditor} timeScheduleEditor the 
   * field editor for the `schedule:TimeScheduleEditor` we're
   * editing
   * @param {Number} mschange the amount we're changing the start/finish
   * time, in milliseconds (e.g. -60000 for "backwards one minute", 
   * 3600000 for "forwards one hour")
   */
  exports.validateTimeScheduleEditor = function (input, dayEditor, timeScheduleEditor, mschange) {
    return timeScheduleEditor.read().then(function (timeSchedule) {
      var slot = input.attr('name').replace('_time', ''),
          //start or finish
      startTime = timeSchedule.get('start'),
          finishTime = timeSchedule.get('finish'),
          startms = startTime.getTimeOfDayMillis(),
          finishms = finishTime.getTimeOfDayMillis(),
          newms;

      function triggerDateBox() {
        var newTime = baja.Time.make(newms);

        newTime.toTimeString({
          textPattern: getBajaTimeFormat() || 'HH:mm',
          ok: function ok(timeStr) {
            setTimeout(function () {
              input.trigger('datebox', {
                method: 'set',
                value: timeStr
              });
            }, 0);
          }
        });
      }

      return exports.validateTimeChange({
        dayEditor: dayEditor,
        startTime: startTime,
        finishTime: finishTime,
        slot: slot,
        mschange: mschange
      }).then(function () {
        newms = slot === 'start' ? startms : finishms;
        newms = (newms + mschange + DAY) % DAY;
        triggerDateBox();
      }, function () {
        newms = getCorrectedTime(dayEditor, startms, finishms, slot, mschange);
        triggerDateBox();
      });
    });
  };

  /**
   * Retrieves or creates an datebox input element inside the given element.
   * If the input does not exist, it will be created inside of a div with
   * class `calendarContainer`. Input will be created in calendar 
   * box mode ('calbox').
   * 
   * @param {JQuery} element element to search for a datebox input, or to
   * create one if it doesn't exist
   * @returns {JQuery} an input element to be dateboxed
   */
  exports.getDateboxInput = function (element) {
    if (element.is(IS_DATEBOX_SELECTOR)) {
      return element;
    }

    var input = element.find(IS_DATEBOX_SELECTOR);

    if (!input.length) {
      var calendarContainer = $('<div class="calendarContainer"/>'),
          calendarContainerInner = $('<div class="calendarContainerInner"/>').appendTo(calendarContainer);

      var options = _.extend(getDateboxDefaultOptions(), {
        mode: 'calbox',
        hideInput: true,
        useInline: true,
        calHighToday: false,
        calHighPick: false,
        themeDateHigh: 'e',
        calNextMonthIcon: 'arrow-r',
        calPrevMonthIcon: 'arrow-l'
      });

      input = $('<input class="hidden" type="date" data-role="datebox" />').attr('data-options', JSON.stringify(options)).appendTo(calendarContainerInner);

      element.html(calendarContainer);
      input.datebox();
    }

    return input;
  };

  /**
   * Finds a datebox input inside the given element, pulls the selected
   * date from it, and converts it to an AbsTime. If no datebox element is
   * found, will return the current time.
   * 
   * @param {JQuery} element the element to search for a datebox input
   * @returns {baja.AbsTime}
   */
  exports.getAbsTime = function (element) {
    var input = exports.getDateboxInput(element),
        jsDate = input.datebox('getTheDate');
    return baja.AbsTime.make({
      jsDate: jsDate,
      offset: jsDate.getTimezoneOffset() * -60000
    });
  };

  return exports;
});
