var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) { return typeof obj; } : function (obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; };

/**
 * @file AOP utilities.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!'], function (baja) {

  "use strict";

  function getFunction(obj, functionName) {
    return baja.iterate(obj, function (obj) {
      return obj[functionName];
    }, function (obj) {
      return obj.prototype;
    });
  }

  function doWrapFunction(obj, methodName, newFunction, wrapFunctions) {
    if (typeof wrapFunctions !== 'function') {
      throw "missing AOP function wrapper";
    }
    var oldFunction, wrappedFunction;

    if (typeof obj === 'function' && typeof methodName === 'function') {
      newFunction = obj;
      oldFunction = methodName;
      wrappedFunction = wrapFunctions(newFunction, oldFunction);

      //ensure we maintain the old prototype in case we're advising a constructor
      wrappedFunction.prototype = oldFunction.prototype;

      return wrappedFunction;
    } else if (((typeof obj === 'undefined' ? 'undefined' : _typeof(obj)) === 'object' || typeof obj === 'function') && typeof methodName === 'string' && typeof newFunction === 'function') {
      oldFunction = getFunction(obj, methodName);
      obj[methodName] = doWrapFunction(newFunction, oldFunction, null, wrapFunctions);
    } else {
      throw "AOP functions must be called with argument signature " + "[Function, Function] or [Object, String, Function]";
    }
  }
  /**
   * Describes behavior that should occur before a particular method on
   * an object is called.
   * 
   * This method may also be called with only two parameters:  `beforeFunc`
   * and `origFunc`. `beforeFunc` should accept a single parameter: an array of
   * arguments that would have ordinarily been passed to `origFunc`.
   * `beforeFunc` should perform any processing of these arguments it needs to,
   * including alteration of the arguments themselves, and return a new array.
   * This array will be passed to `origFunc` using `apply()`. If `beforeFunc`
   * returns `false`, then `afterFunc` will not be called at all and 
   * `undefined` will be returned.
   * 
   * @memberOf niagara.util.aop
   * @param {Object} obj an object containing the method to apply the aspect
   * to. If this object is a prototype then every object created with the
   * prototype will receive this new behavior.
   * 
   * @param {String} methodName the name of the method to apply the aspect to
   * 
   * @param {Function} beforeFunction the function to execute before the
   * target function. `beforeFunction` can accept a single parameter which is
   * an array containing the arguments passed to the target function.
   * `beforeFunction` has the option of returning a new array of arguments
   * which will then be passed into the target function in place of the
   * original arguments; if nothing is returned, the original arguments array
   * will be used. Please note that in either case, any modifications made to
   * the arguments array within `beforeFunction` will still pass through into
   * the target function - however they will NOT pass through into any AOP
   * functions registered using `after`. `beforeFunction` may also return
   * `false` to cancel the execution of the target function.
   */
  function before(obj, methodName, beforeFunction) {
    return doWrapFunction(obj, methodName, beforeFunction, function (newFunc, oldFunc) {
      return function () {
        var args = Array.prototype.slice.call(arguments),
            newArgs = newFunc.call(this, args);
        if (newArgs !== false) {
          return oldFunc.apply(this, newArgs || args);
        }
      };
    });
  }

  /**
   * Describes behavior that should occur after a particular method on
   * an object is called.
   * 
   * This method may also be called with only two parameters: `func1` and
   * `func2`. `func2` should take two parameters: `args`, an array containing
   * the arguments passed to `func1`, and `value`, the value returned from
   * `func1`. `after` will return a new function that runs `func1` and `func2`
   * in sequence, returning the value returned from `func2`. If func2 returns
   * `undefined`, then the original value from func1 will be returned in its
   * stead - so if you wish to replace a defined value with a nullish one,
   * return `null` from func2.
   * 
   * 
   * @memberOf niagara.util.aop
   * @param {Object} obj an object containing the method to apply the aspect
   * to. If this object is a prototype then every object created with the
   * prototype will receive this new behavior.
   * 
   * @param {String} methodName the name of the method to apply the aspect to
   * 
   * @param {Function} afterFunction the function to execute after the
   * target function. `afterFunction` can accept two parameters: the first
   * consisting of the arguments array passed to the target function, and the
   * second consisting of the target function's return value. The value
   * returned by `afterFunction` will replace the one returned by the target
   * function.
   */
  function after(obj, methodName, afterFunction) {
    return doWrapFunction(obj, methodName, afterFunction, function (newFunc, oldFunc) {
      return function () {
        var args = Array.prototype.slice.call(arguments),
            value = oldFunc.apply(this, args),
            newResult = newFunc.call(this, args, value);
        if (newResult === undefined) {
          return value;
        } else {
          return newResult;
        }
      };
    });
  }

  /**
   * Convenience method allowing you to add before/after AOP advice to
   * multiple functions on an object in one statement.
   * 
   * @memberOf niagara.util.aop
   * 
   * @param {Object} obj an object requiring behavior advice
   * @param {Object} advice an object containing functions to applied to `obj`
   * using `before()` and `after`
   * @param {Object} [advice.before] a mapping from method name to `before()`
   * advice functions
   * @param {Object} [advice.after] a mapping from method name to `after()`
   * advice functions
   * @param {Object} [advice.toCallback] a mapping from method name to
   * `toCallback()` advice functions
    * 
   * @returns {Object} the newly advised object
   */
  function advise(obj, advice) {
    baja.strictArg(obj, Object);

    advice = baja.objectify(advice);

    var befores = advice.before || {},
        afters = advice.after || {};
    baja.iterate(befores, function (func, funcName) {
      before(obj, funcName, func);
    });
    baja.iterate(afters, function (func, funcName) {
      after(obj, funcName, func);
    });

    return obj;
  }

  /**
   * Performs `advise` on the given function's prototype. Very useful for
   * allowing one constructor to extend another.
   * 
   * @memberOf niagara.util.aop
   * 
   * @param {Function} func a function whose prototype to advise 
   * @param {Object} advice an object containing AOP before/after functions
   * (see `advise()`)
   * @returns {Function} the newly advised function
   */
  function advisePrototype(func, advice) {
    baja.strictArg(func, Function);
    advise(func.prototype, advice);
    return func;
  }

  /** 
   * @namespace
   * @name niagara.util.aop
   */
  return {
    advise: advise,
    advisePrototype: advisePrototype,
    after: after,
    before: before
  };
});
