/**
 * @file A view for constructing a JQM list.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * @private
 * @module mobile/util/mobile/views/ListView
 */
define(['baja!', 'jquery', 'jquerymobile', 'Promise', 'underscore', 'mobile/util/slot', 'mobile/util/ord', 'mobile/util/mobile/mobile', 'mobile/util/mobile/views/MobileView', 'bajaux/icon/iconUtils', 'bajaux/mixin/subscriberMixIn'], function (baja, $, jqm, Promise, _, slotUtil, ordUtil, mobileUtil, MobileView, iconUtils, subscribable) {

  "use strict";

  var encodePageId = mobileUtil.encodePageId,
      decodePageId = mobileUtil.decodePageId,
      getActivePage = mobileUtil.getActivePage,
      toHtml = iconUtils.toHtml;

  /**
   * Check to see if clicking this property in the property sheet should link
   * new a new property sheet for this property.
   * 
   * @param {baja.Property} prop property to check for linkability
   * @returns {Boolean} true if this is a subcomponent we can show on its own
   * property sheet
   */

  function isLinkable(prop) {
    return prop.getType().isComplex();
  }

  /**
   * A Complex list view widget, displaying component slots in the form of
   * a JQM `listview`. Supports dynamic listing of component properties and
   * inline expansion of properties.
   * 
   * Note that only Property slots will be displayed - Actions and Topics
   * will not.
   * 
   * @class
   * @alias module:mobile/util/mobile/views/ListView
   * @extends module:mobile/util/mobile/views/MobileView
   */
  var ListView = function ListView() {
    MobileView.apply(this, arguments);
    subscribable(this);
  };
  ListView.prototype = Object.create(MobileView.prototype);
  ListView.prototype.constructor = ListView;

  /**
   * A function used to restrict which slots are displayed in the list view.
   * By default, only returns slots which are Properties.
   * 
   * @param {baja.Slot} slot 
   * @returns {Boolean} true if the given slot should be included.
   */
  ListView.prototype.shouldIncludeSlot = function shouldIncludeSlot(slot) {
    return slot.isProperty() && !slot.isAction() && !slot.isTopic();
  };

  /**
   * Arms a handler on expandable slots (slots for which `makeListItemParams`
   * returns `expandable === true`) that will call `this.doExpand` when they
   * are clicked.
   * @param {jQuery} element
   */
  ListView.prototype.doInitialize = function (element) {
    var that = this;

    return MobileView.prototype.doInitialize.call(this, element).then(function () {
      element.on('click', 'a.expandable', function () {
        var slotName = $(this).children('span.slotName').text();

        that.doExpand(slotName).catch(baja.error);

        return false;
      });

      element.on('click', 'a.link', function (e) {
        var a = $(this),
            slotName = a.parents('li.property').attr('id');
        that.linkClicked(decodePageId(slotName), a);
      });
    });
  };

  /**
   * Loads a Complex into the listview, building up the HTML necessary to
   * display its slots. Delegates the actual construction of the `<ul>` to
   * `makeListView`.
   * 
   * @param {baja.Complex} complex the Complex value to be displayed in the list
   */
  ListView.prototype.doLoad = function doLoad(complex) {
    var _this = this;

    baja.strictArg(complex, baja.Complex);

    var highlightedSlot = this.$highlightedSlot;

    return this.makeListView(complex).then(function (ul) {
      _this.$ul = ul;
      _this.jq().empty().append(ul).enhanceWithin();
      if (highlightedSlot) {
        _this.highlight(highlightedSlot);
      }
    });
  };

  /**
   * When expanding an entry in a ListView, this will scroll the screen far
   * enough that the entire expanded div will be in view (but not so far that
   * the list entry itself will be scrolled off the top of the screen). This
   * will be called from `doExpand()`.
   * 
   * @private
   * @param {jQuery} expandedDiv the div that has just been expanded
   * @see #doExpand
   */
  ListView.prototype.scrollExpandedDivIntoView = function scrollExpandedDivIntoView(expandedDiv) {
    //after we expand out the slot, let's check to see if the expanded
    //div goes past the bottom of our viewable area - and if it does,
    //scroll down a bit so we can see it all
    var activePage = getActivePage(),
        divBottom = expandedDiv.offset().top + expandedDiv.height(),
        headerHeight = activePage.children(':jqmData(role=header)').height(),
        footerHeight = activePage.children(':jqmData(role=footer)').height(),
        $window = $(window),
        scrollTop = $window.scrollTop(),


    //the bottom of the content area / top edge of the footer bar
    scrollBottom = scrollTop + $window.height() - footerHeight;

    if (divBottom > scrollBottom) {
      //do the scroll
      //don't autoscroll the property li off the top of the screen
      var maxTop = expandedDiv.prev().offset().top - headerHeight;
      jqm.silentScroll(Math.min(maxTop, scrollTop + (divBottom - scrollBottom)));
    }
  };

  /**
   * Upon clicking a component property, will load up a div (populated using
   * the override hook `populateExpandedDiv`) and display it inline after the
   * `<li>` for the selected slot. Calling this function a second time will
   * hide that div.
   * 
   * @param {String|baja.Slot} slotName the property slot to show an expansion
   * for
   * @returns {Promise} a promise to be resolved after the jQuery show/hide
   * animation is complete
   */
  ListView.prototype.doExpand = function doExpand(slotName) {
    var _this2 = this;

    var toCarat = function toCarat(dir) {
      return 'ui-icon-carat-' + dir;
    };

    var slot = this.value().getSlot(slotName),
        slotId = encodePageId(slot.getName()),
        li = this.$ul.children('li#' + slotId),
        expandable = li.children('.expandable').removeClass('udrl'.split('').map(toCarat).join(' ')),
        expandedDiv = li.next('.expanded');

    return new Promise(function (resolve) {
      var after = function after(dir) {
        expandable.addClass(toCarat(dir));
        _this2.scrollExpandedDivIntoView(expandedDiv);
        resolve();
      };

      if (expandedDiv.length) {
        if (expandedDiv.is(':visible')) {
          expandedDiv.hide('fast', function () {
            return after('d');
          });
        } else {
          expandedDiv.show('fast', function () {
            return after('u');
          });
        }
      } else {
        expandedDiv = $('<div class="expanded"/>').insertAfter(li).hide();
        _this2.populateExpandedDiv(slot, expandedDiv);
        expandedDiv.show('fast', function () {
          return after('u');
        });
      }
    });
  };

  /**
   * Highlights the selected property in blue (or whatever color is dictated 
   * by the JQM theme).
   * 
   * @param {String|baja.Property} prop the name of the property to highlight
   */
  ListView.prototype.highlight = function highlight(prop) {
    this.removeHighlight();

    var ul = this.$ul,
        pageId = encodePageId(prop);
    this.$highlightedSlot = prop;
    ul.children('li#' + pageId).addClass('ui-btn-active');
  };

  /**
   * Removes highlighting from all list items.
   */
  ListView.prototype.removeHighlight = function () {
    delete this.$highlightedSlot;
    this.$ul.children().removeClass('ui-btn-active');
  };

  /**
   * Executed whenever a linkable list item is clicked.
   * @param {baja.Property} prop
   * @param {jQuery} a
   */
  ListView.prototype.linkClicked = function linkClicked(prop, a) {
    this.highlight(prop);
  };

  /**
   * Given an expanded property, populate the expanded div with any information,
   * form elements, etc. that should be shown. Default behavior is to do
   * nothing (and may be left that way if you do not plan to do any inline
   * property expansion in your view).
   * 
   * @param {baja.Property} prop the property being expanded
   * @param {jQuery} expandedDiv the div that is expanded to show property
   * details
   */
  ListView.prototype.populateExpandedDiv = function populateExpandedDiv(prop, expandedDiv) {};

  /**
   * Creates the `<ul>` that contains the list of all this component's
   * properties. Each item is built using `makeListItem()`. Note that the
   * output object is raw HTML and has not yet been enhanced with jQuery Mobile
   * widgets.
   * 
   * @param {baja.Complex} complex the child component of the view's main
   * component that we are creating a list item for
   * @returns {Promise} promise to be resolved with a created `<ul>` jQuery
   * object
   */
  ListView.prototype.makeListView = function makeListView(complex) {
    var _this3 = this;

    var slots = complex.getSlots(function (slot) {
      return _this3.shouldIncludeSlot(slot) && !slotUtil.isHidden(slot);
    }).toArray(),
        makeListItems = _.map(slots, function (slot) {
      return _this3.makeListItem(complex, slot);
    });

    return Promise.all(makeListItems).then(function (items) {
      return $('<ul data-role="listview" data-theme="c" />').html(items);
    });
  };

  /**
   * Adds icons to a `<li>` element for display. Any existing icons will be
   * removed and replaced with the new ones.
   * 
   * @param {jQuery} li the listview item to add icons to
   * @param {baja.Icon} icon the icon from a Baja component
   */
  ListView.prototype.prependIcons = function prependIcons(li, icon) {
    if (!icon) {
      return Promise.resolve();
    }

    return toHtml(icon).then(function (html) {
      return li.find('.bajaImage').html(html);
    });
  };

  /**
   * Creates an `<li>` element to be added to the view's property list.
   * 
   * @param {baja.Complex} complex the component being displayed in this
   * view (this will usually just be `this.value()` - but not always, we may
   * want to custom-create list items from the output of a server-side call,
   * etc)
   * @param {baja.Property} prop the property on `component` we are creating a
   * list item for
   */
  ListView.prototype.makeListItem = function makeListItem(complex, prop) {
    var _this4 = this;

    var li = $('<li class="property" data-icon="carat-r"/>').attr('id', encodePageId(String(prop))).append('<span class="bajaImage"></span>');

    return this.makeListItemParams(complex, prop).then(function (params) {
      return _this4.prependIcons(li, params.icon).then(function () {
        $('<label class="title" />') //label with property name
        .text(params.title || '').appendTo(li);

        //  holds the original slot name - hold on to a copy for use when handling
        //  slot renames
        $('<span class="hidden slotName" />').text(String(prop)).appendTo(li);

        //display output of this slot
        $('<div class="display" />').text(params.display || '').appendTo(li);

        if (params.linkable || params.expandable) {
          var a = $('<a />');
          a.jqmData('ord', params.ord);

          if (params.expandable) {
            a.addClass('expandable');
            li.attr('data-icon', 'carat-d');
          } else {
            a.addClass('link');
          }
          li.wrapInner(a);
        }

        return li;
      });
    });
  };

  /**
   * Creates the necessary parameters to build up a `<li>` element for a given
   * property. The returned object can include the following properties:
   * 
   * - `title`: the string title (commonly the property display name)
   * - `display`: the display value to show on the right hand side, if any
   * - `icon`: a `baja.Icon` icon to display for this property
   * - `expandable`: true if this list item should respond to a click by
   *   expanding a div below it (populated using `populateExpandedDiv()`)
   * - `linkable`: true if this property should respond to a click by linking
   *   to a subcomponent
   * - `ord`: the ORD (baja.Ord or String) that this list item should link to -
   *   only used if `linkable` is also true
   * 
   * These flags will cause the `<a>` tags within the `<li>` to be given CSS
   * classes of `expandable` or `link`. Basic click handlers will be added by
   * default by `PageViewManager.armHandlers()` - any additional functionality
   * should be handled by your own application.
   * 
   * @param {baja.Complex} complex the component whose property is displayed
   * in the list item (the same one passed to `makeListItem`)
   * @param {baja.Property} prop the property on `component` being displayed
   * @returns {Promise} promise to be resolved with an object literal with
   * parameters for the list item
   */
  ListView.prototype.makeListItemParams = function makeListItemParams(complex, prop) {
    var subComponent = complex.get(prop),
        ord = ordUtil.deriveOrd(subComponent);

    if (prop.getType().is('baja:VirtualGateway')) {
      ord += '|virtual:/';
    }

    return Promise.resolve({
      ord: ord,
      title: complex.getDisplayName(prop),
      display: baja.SlotPath.unescape(complex.getDisplay(prop)),
      icon: subComponent.getIcon(),
      expandable: slotUtil.isEditable(prop),
      linkable: isLinkable(prop) && !slotUtil.isReadonly(prop)
    });
  };

  /**
   * Updates the display value on the `<li>` in this view for the specified
   * property.
   * 
   * @param {baja.Property} prop the property to be updated
   * @returns {Promise}
   */
  ListView.prototype.updateDisplay = function updateDisplay(prop) {
    if (!this.$ul) {
      return Promise.resolve();
    }

    var id = encodePageId(String(prop)),
        li = this.$ul.find('li.property#' + id),
        displayDiv = li.find('div.display'),
        value = this.value(),
        child = value.get(prop);

    if (displayDiv.length) {
      displayDiv.text(value.getDisplay(prop));
    }

    return Promise.resolve(child && child.getType().isComponent() && this.prependIcons(li, child.getIcon()));
  };

  /**
   * Arms ListView with additional responses to Baja events. Keeps list items
   * updated on `changed` events, adjusts display titles and ORD links on
   * `renamed` events, and disables list items on `removed` events.
   */
  ListView.prototype.getSubscriber = function getSubscriber() {
    var _this5 = this;

    var sub = this.$subscriber;

    if (sub) {
      return sub;
    }

    sub = this.$subscriber = new baja.Subscriber();

    sub.attach('changed', function (prop) {
      return _this5.updateDisplay(prop);
    });

    sub.attach('renamed', function (prop, oldName) {
      var propId = encodePageId(prop.getName()),
          oldPropId = encodePageId(oldName),
          li = _this5.$ul.children('li.property#' + oldPropId),
          value = _this5.value(),
          newOrd = ordUtil.deriveOrd(value.get(prop));

      li.find('label.title').addClass('renamed').text(value.getDisplayName(prop));
      li.find('span.slotName').text(prop.getName());
      li.find('a.link').jqmData('ord', newOrd);
      li.attr('id', propId);
    });

    sub.attach('removed', function (prop) {
      var propId = encodePageId(prop.getName()),
          li = _this5.$ul.children('li.property#' + propId);

      li.find('a').removeClass('link expandable');
      li.find('label.title').addClass('removed').append(' (removed)');
    });

    sub.attach('added', function (prop) {
      var value = _this5.value();

      _this5.makeListItem(value, prop).then(function (li) {
        return _this5.$ul.append(li).listview('refresh');
      }).catch(baja.error);
    });

    return sub;
  };

  return ListView;
});
