/**
 * @file ORD-related utilities.
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*global unescape */

define(['baja!'], function (baja) {

  "use strict";

  /**
   * If the given ord ends with a slash, remove it - unless the ord ends with
   * a colon (e.g. 'station:|slot:') which can cause some problems with 
   * ord resolution. In this case, ADD a slash.
   * 
   * @private
   * @inner
   * @memberOf niagara.util.ord
   * @param {String|baja.Ord} ord
   * @returns {String} the ord with its trailing '/' removed, or a trailing
   * ':' changed to ':/'
   */

  function chopLastSlash(ord) {
    if (!ord) {
      return "";
    }

    ord = String(ord);
    var len = ord.length,
        lastChar = ord.substring(len - 1, len),
        nextToLastChar = ord.substring(len - 2, len - 1);

    if (lastChar === '/' && nextToLastChar !== ':') {
      ord = ord.substring(0, len - 1);
    } else if (lastChar === ':') {
      ord = ord + '/';
    }

    return ord;
  }

  /**
   * Convenience method for retrieving a component from a given ORD. 
   * 
   * @memberOf niagara.util.ord
   * @param {baja.Ord|String} ord the ord to resolve
   * @param {Boolean} [lease] whether or not the retrieved component should
   * be leased (defaults to true).
   * @returns {Promise} promise to be resolved with the object retrieved
   */
  function get(ord, lease) {
    ord = baja.Ord.make(ord);

    if (lease === undefined) {
      lease = true;
    }

    return ord.get({ lease: lease });
  }

  /**
   * Converts an ORD into an array of all its sub-ORDs, beginning at the root. 
   * For example, the ORD `'a/b/c'` would convert to `['a', 'a/b', 'a/b/c']`.
   * 
   * @memberOf niagara.util.ord
   * @param ord {String|baja.Ord} ord the ORD to convert to sub-ORDs
   * @returns {Array} an array of all the given ORD's sub-ORDs
   */
  function makeOrdHistory(ord) {
    ord = unescape(String(ord));

    //ensure there is no trailing slash as this will add one too many 
    //matches to the split below. chopLastSlash will manage whether to
    //include one or not
    if (ord.match(/\/$/)) {
      ord = ord.substring(0, ord.length - 1);
    }

    var split = ord.split('/'),
        i = 0,
        history = [];
    for (i = 0; i < split.length; i++) {
      history.push(chopLastSlash(split.slice(0, i + 1).join('/')));
    }
    return history;
  }

  /**
   * Checks two ORDs for equivalence. ORDs are considered equivalent if 
   * their string representations, omitting any trailing slashes, are equal.
   * 
   * @memberOf niagara.util.ord
   * @param {baja.Ord|String} ord1
   * @param {baja.Ord|String} ord2
   */
  function equivalent(ord1, ord2) {
    if (!ord1 || !ord2) {
      return false;
    }

    return baja.Ord.make(ord1).relativizeToSession().equals(baja.Ord.make(ord2).relativizeToSession());
  }

  /**
   * Calculates the ORD for the given `Complex` object. If the object is a
   * `baja.Component`, `getNavOrd()` will be returned. If the object is
   * otherwise a `baja.Complex` (e.g. in case of a `Struct`), the ORD will be
   * constructed by appending property names from the nearest `baja.Component`
   * parent.
   * 
   * @memberOf niagara.util.ord
   * @param {baja.Complex} complex the object whose ORD we wish to know
   * @returns {String} a String representation of the ORD
   */
  function deriveOrd(complex) {
    if (complex) {
      if (complex instanceof baja.Ord || typeof complex === 'string') {
        return complex.toString();
      }

      if (complex.getType().isComponent() && complex.isMounted()) {
        return complex.getNavOrd().relativizeToSession().toString();
      }

      if (complex.getType().isComplex()) {
        var parentOrd = deriveOrd(complex.getParent());
        if (parentOrd) {
          return parentOrd + '/' + complex.getPropertyInParent();
        }
      }
    }
    return undefined;
  }

  /**
   * @namespace
   * @name niagara.util.ord
   */
  return {
    chopLastSlash: chopLastSlash,
    deriveOrd: deriveOrd,
    equivalent: equivalent,
    get: get,
    makeOrdHistory: makeOrdHistory
  };
});
