/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Jeremy Narron
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/array/ArrayCompositeBuilder
 */
define(['underscore', 'Promise', 'nmodule/webEditors/rc/fe/config/CompositeBuilder'], function (_, Promise, CompositeBuilder) {
  'use strict';

  /**
   * A `CompositeBuilder` subclass that uses an Array as the data source.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/config/CompositeBuilder
   * @alias module:nmodule/webEditors/rc/fe/array/ArrayCompositeBuilder
   * @param {Object} params
   * @param {Array} params.dataSource specify an array data source for this
   * builder.
   */
  var ArrayCompositeBuilder = function ArrayCompositeBuilder(params) {
    CompositeBuilder.apply(this, arguments);
  };
  ArrayCompositeBuilder.prototype = Object.create(CompositeBuilder.prototype);
  ArrayCompositeBuilder.prototype.constructor = ArrayCompositeBuilder;

  /**
   * Maps each array index to a key.
   *
   * @returns {Array.<String>} each index in the Array, as a String
   */
  ArrayCompositeBuilder.prototype.getKeys = function getKeys() {
    return _.map(this.getDataSource(), function (val, i) {
      return String(i);
    });
  };

  /**
   * Pass in an index to get the value at that index.
   *
   * @param {String} index
   * @returns {*} the value at the given index
   */
  ArrayCompositeBuilder.prototype.getValueFor = function getValueFor(index) {
    return this.getDataSource()[index];
  };

  /**
   * Override config object with mini formFactor for each field editor
   *
   * @returns {Object}
   */
  ArrayCompositeBuilder.prototype.getConfigFor = function getConfigFor() {
    return {
      formFactor: 'mini'
    };
  };

  /**
   * Sets the data source for this builder.
   *
   * @param {Array} dataSource
   * @returns {Promise}
   */
  ArrayCompositeBuilder.prototype.setDataSource = function setDataSource(dataSource) {
    if (!Array.isArray(dataSource)) {
      return Promise.reject(new Error('Array required'));
    }
    return CompositeBuilder.prototype.setDataSource.apply(this, arguments);
  };
  return ArrayCompositeBuilder;
});
