/**
 * @copyright 2022 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/DropDownEditor
 */
define(['baja!', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/baja/BaseEditor'], function (baja, Promise, _, BaseEditor) {
  'use strict';

  /**
   * A field editor for picking from a list of strings.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/DropDownEditor
   * @since Niagara 4.13
   */
  var DropDownEditor = function DropDownEditor(params) {
    BaseEditor.call(this, _.extend({
      keyName: 'DropDownEditor'
    }, params));
  };
  DropDownEditor.prototype = Object.create(BaseEditor.prototype);
  DropDownEditor.prototype.constructor = DropDownEditor;

  /**
   * Initializes this editor.
   *
   * @param {JQuery} dom
   */
  DropDownEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.addClass('DropDownEditor');
    dom.html('<select class="item-selection" />');
    dom.on('change', '.item-selection', function (e) {
      that.setModified(true);
      e.preventDefault();
      return false;
    });
  };

  /**
   * Loads a list of strings, that are presented in a dropdown.
   *
   * @param {String} selected list of strings to present.
   * @returns {Promise}
   */
  DropDownEditor.prototype.doLoad = function (selected) {
    return this.$populateSelectOptions(selected);
  };

  /**
   * Reads the select element and return the currently selected value.
   *
   * @returns {String} the selected property name else an empty string.
   */
  DropDownEditor.prototype.doRead = function () {
    var selectedProperty = this.$getSelect().val();
    if (selectedProperty) {
      return selectedProperty;
    }
    return "";
  };

  /**
   * Enables or disables the select dropdown.
   *
   * @param {Boolean} enabled
   */
  DropDownEditor.prototype.doEnabled = function (enabled) {
    this.$getSelect().prop('disabled', this.isReadonly() || !enabled);
  };

  /**
   * Disables or enables the select dropdown.
   *
   * @param {Boolean} readonly
   */
  DropDownEditor.prototype.doReadonly = function (readonly) {
    this.$getSelect().prop('disabled', !this.isEnabled() || readonly);
  };

  /**
   * Clean up after this editor
   */
  DropDownEditor.prototype.doDestroy = function () {
    this.jq().removeClass('DropDownEditor');
  };

  /**
   * Populates the dropdown with the string options, selecting the value which matches
   * the current selection.
   *
   * @private
   * @param {String} selected the current load value to select in the list if it exists.
   * @returns {Promise}
   */
  DropDownEditor.prototype.$populateSelectOptions = function (selected) {
    var that = this,
      selectElement = that.$getSelect();

    // clear out the dropdown options
    that.$getSelect().empty();

    // add a blank element to use if no values,
    // this element will be removed it the current value is an option.
    selectElement.append('<option class="ux-option blank-option" value=""></option>');

    // and make it the selected one.
    selectElement.val('');
    return that.supplyItems(selected).then(function (items) {
      if (items) {
        items.forEach(function (item) {
          if (item) {
            selectElement.append('<option class="ux-option" value="' + _.escape(item) + '">' + _.escape(baja.SlotPath.unescape(item)) + '</option>');
            if (selected === item) {
              // if this matches the current value, set it as the selected option
              selectElement.val(item);
              // remove the blank option
              that.jq().find('.blank-option').remove();
            }
          }
        });
      }
    });
  };

  /**
   * Sub-classes override this function to supply the list of string options to display.
   * @returns {Promise} a promise returning the list of string options to display.
   */
  DropDownEditor.prototype.supplyItems = function () {
    return Promise.resolve([]);
  };

  /**
   * Get the dropdown select element.
   *
   * @private
   * @returns {JQuery}
   */
  DropDownEditor.prototype.$getSelect = function () {
    return this.jq().find('.item-selection');
  };
  return DropDownEditor;
});
