/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/OverrideRelTimeEditor
 */
define(['baja!', 'log!nmodule.webEditors.rc.fe.baja.OverrideRelTimeEditor', 'underscore', 'bajaux/events', 'bajaux/Properties', 'nmodule/js/rc/switchboard/switchboard', 'nmodule/webEditors/rc/fe/CompositeEditor', 'nmodule/webEditors/rc/fe/baja/OrderedMapEditor', 'nmodule/webEditors/rc/fe/config/CompositeBuilder'], function (baja, log, _, events, Properties, switchboard, CompositeEditor, OrderedMapEditor, CompositeBuilder) {
  'use strict';

  var MODIFY_EVENT = events.MODIFY_EVENT,
    logError = log.severe.bind(log),
    CUSTOM_DURATION = -1,
    DURATION_KEY = 'durationSelect',
    RELTIME_KEY = 'relTime';

  //these display values get processed through the `control` lexicon
  var DURATIONS = [['override.perm', 0], ['override.min1', 60000], ['override.min15', 15 * 60000], ['override.min30', 30 * 60000], ['override.hour1', 60 * 60000], ['override.hour2', 120 * 60000], ['override.hour3', 180 * 60000], ['override.custom', CUSTOM_DURATION]];

  /**
   * Get the `OrderedMap` of preset override durations. It will be dynamically
   * generated each time to only those less than or equal to `relTime`.
   *
   * @private
   * @param {baja.RelTime} relTime
   * @param {Array} durations
   * @returns {baja.OrderedMap} `OrderedMap` to be loaded into the dropdown
   * editor
   */
  function getDurationMap(relTime, durations) {
    if (!baja.hasType(relTime, 'baja:RelTime')) {
      relTime = null;
    }
    var map = new baja.OrderedMap();
    _.each(durations, function (entry) {
      var key = entry[0],
        duration = entry[1];
      if (relTime === null || duration === CUSTOM_DURATION || duration !== 0 && duration <= relTime.valueOf()) {
        map.put(key, duration);
      }
    });
    return map;
  }

  /**
   * A field editor for working with a RelTime on a `control:Override`. Provides
   * a select dropdown for predetermined override durations as well as entering
   * custom durations.
   *
   * This editor supports the following Properties:
   *
   * - `min`: set a `baja.RelTime` value to set the min override time.
   * - `max`: set a `baja.RelTime` value to set the max override time. If given,
   *   it will also be shown at load time.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/OverrideRelTimeEditor
   */
  var OverrideRelTimeEditor = function OverrideRelTimeEditor() {
    var that = this;
    CompositeEditor.apply(that, arguments);
    that.validators().add(function () {
      return that.getChildEditors().validateAll();
    });

    //NCCB-29820
    //if a new value has been selected from the duration dropdown, wait for
    //the new value to load into the RelTime editor before allowing a read
    switchboard(that, {
      read: {
        notWhile: '$updateFromDuration'
      }
    });
  };
  OverrideRelTimeEditor.prototype = Object.create(CompositeEditor.prototype);
  OverrideRelTimeEditor.prototype.constructor = OverrideRelTimeEditor;
  OverrideRelTimeEditor.prototype.makeBuilder = function () {
    var that = this,
      builder = new CompositeBuilder();
    builder.getKeys = function () {
      return [DURATION_KEY, RELTIME_KEY];
    };
    builder.getValueFor = function (key) {
      switch (key) {
        case DURATION_KEY:
          return getDurationMap(that.$getMax(), that.$getDurations());
        case RELTIME_KEY:
          return that.$getMax() || this.getDataSource();
      }
    };
    builder.getConfigFor = function (key) {
      switch (key) {
        case DURATION_KEY:
          return {
            type: OrderedMapEditor,
            properties: {
              lexicon: that.$getLexiconName()
            },
            formFactor: 'mini'
          };
        case RELTIME_KEY:
          return {
            properties: _.extend(that.properties().subset(['min', 'max', 'showDay']).toObject(), {
              showMilliseconds: true
            }),
            formFactor: 'mini'
          };
      }
    };
    builder.getDomFor = function (key) {
      return that.jq().children('.js-' + key);
    };
    return builder;
  };

  /**
   * Determine the `max` for this editor.
   *
   * @returns {baja.RelTime|null} the max override duration RelTime, or null if
   * no max override specified
   */
  OverrideRelTimeEditor.prototype.$getMax = function () {
    var max = this.properties().getValue('max');
    return baja.hasType(max, 'baja:RelTime') ? max : null;
  };

  /**
   * Return the durations to use for this editor.
   * @returns {Array} the durations to use for this editor
   */
  OverrideRelTimeEditor.prototype.$getDurations = function () {
    return DURATIONS;
  };

  /**
   * Return the lexicon module name.
   * @returns {string}
   */
  OverrideRelTimeEditor.prototype.$getLexiconName = function () {
    return "control";
  };

  /**
   * When a new duration is selected, update the RelTime editor. It should
   * enable itself if a custom duration is selected, otherwise be loaded
   * with the selected duration and disabled.
   *
   * @private
   * @returns {Promise} promise to be resolved after the RelTime editor has
   * enabled/loaded as appropriate
   */
  OverrideRelTimeEditor.prototype.$updateFromDuration = function () {
    var durationEd = this.$getDurationEditor(),
      relTimeEd = this.$getRelTimeEditor();
    return durationEd.read().then(function (duration) {
      if (duration === CUSTOM_DURATION) {
        return relTimeEd.setEnabled(true);
      } else {
        return relTimeEd.load(baja.RelTime.make(duration)).then(function () {
          return relTimeEd.setEnabled(false);
        });
      }
    });
  };

  /**
   * Initializes an `OrderedMapEditor` and `RelTimeEditor` for the dropdown of
   * preset durations and custom duration, respectively.
   *
   * Arms event handlers to ensure that when the dropdown changes, the
   * `RelTimeEditor` is enabled/disabled/loaded as appropriate.
   *
   * @param {JQuery} dom
   * @returns {Promise} promise to be resolved when the `OrderedMap`
   * and `RelTime` editors are initialized
   */
  OverrideRelTimeEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.html('<span class="js-' + DURATION_KEY + '"/>' + '<span class="js-' + RELTIME_KEY + '"/>');

    //when a new preset duration is selected...
    dom.on(MODIFY_EVENT, '.js-' + DURATION_KEY, function () {
      that.$updateFromDuration()["catch"](logError);
    });
    return CompositeEditor.prototype.doInitialize.apply(this, arguments);
  };

  /**
   * After loading in a `RelTime`, update the value selected in the dropdown
   * to match.
   *
   * @param {baja.RelTime} relTime
   * @returns {Promise}
   */
  OverrideRelTimeEditor.prototype.doLoad = function (relTime) {
    var that = this;
    return CompositeEditor.prototype.doLoad.apply(that, arguments).then(function () {
      // if the duration hasn't been set, and there's a maxDuration, use that.
      var durationToUse = that.$getMax() && relTime === baja.RelTime.DEFAULT ? that.$getMax() : relTime,
        durationEd = that.$getDurationEditor();
      var millis = durationToUse.getMillis();
      if (!durationEd.getKeysFor(millis).length) {
        millis = CUSTOM_DURATION;
      }
      return durationEd.setSelectedValue(millis);
    });
  };

  /**
   * Returns the current value of the `RelTimeEditor`.
   *
   * @returns {Promise} promise to be resolved with the currently entered
   * `baja.RelTime`
   */
  OverrideRelTimeEditor.prototype.doRead = function () {
    return this.$getRelTimeEditor().read();
  };

  /**
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/RelTimeEditor}
   */
  OverrideRelTimeEditor.prototype.$getRelTimeEditor = function () {
    // noinspection JSValidateTypes
    return this.getBuilder().getEditorFor(RELTIME_KEY);
  };

  /**
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/OrderedMapEditor}
   */
  OverrideRelTimeEditor.prototype.$getDurationEditor = function () {
    // noinspection JSValidateTypes
    return this.getBuilder().getEditorFor(DURATION_KEY);
  };
  return OverrideRelTimeEditor;
});
