/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/util/DepthSubscriber
 */
define(['baja!', 'log!nmodule.webEditors.rc.fe.baja.util.DepthSubscriber', 'Promise', 'underscore'], function (baja, log, Promise, _) {
  'use strict';

  var difference = _.difference,
    flatten = _.flatten,
    invoke = _.invoke,
    union = _.union;
  var logError = log.severe.bind(log);

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  function isReadableComponent(comp) {
    // NCCB-31314: sometimes a visible component is not actually readable
    return baja.hasType(comp, 'baja:Component') && comp.getPermissions().hasOperatorRead();
  }

  /**
   * Convert the value to an array if it isn't one already.
   * @inner
   * @param {Array|*} arr
   * @returns {Array} the given array, or a single-element array containing the
   * input value
   */
  function toArray(arr) {
    return Array.isArray(arr) ? arr : [arr];
  }

  /**
   * Load child components of the given array of components.
   * @inner
   * @param {Object} params the Object Literal for the method's arguments.
   * @param {Array.<baja.Component>} params.comps
   * @param {Function} [params.subscriptionFilter]
   * @param {Number} [params.depth]
   * @param {Number} [params.originalDepth]
   * @returns {Promise} promise to be resolved with a flattened array of
   * all child components of the given components
   */
  function getAllKids(_ref) {
    var comps = _ref.comps,
      subscriptionFilter = _ref.subscriptionFilter,
      depth = _ref.depth,
      originalDepth = _ref.originalDepth;
    return Promise.all(invoke(comps, 'loadSlots')).then(function () {
      return flatten(comps.map(function (comp) {
        var kids = comp.getSlots().properties().filter(function (slot) {
          return comp.get(slot).getType().isComponent();
        }).toValueArray();
        if (subscriptionFilter) {
          return kids.filter(function (component) {
            return subscriptionFilter({
              component: component,
              depth: originalDepth - depth + 1
            });
          });
        }
        return kids;
      }));
    });
  }

  /**
   * Subscribe the entire component tree to the given depth.
   *
   * @inner
   * @param {Object} params the Object Literal for the method's arguments.
   * @param {Array.<baja.Component>} params.comps components to subscribe
   * @param {baja.Subscriber} params.sub the subscriber to use
   * @param {Number} params.depth the depth to subscribe - 0 subscribes nothing, 1
   * subscribes only the given components, 2 subscribes components and their
   * kids, etc.
   * @param {module:nmodule/webEditors/rc/wb/util/subscriptionUtil~SubscriptionFilter} [params.subscriptionFilter]
   * @param {module:nmodule/webEditors/rc/wb/util/subscriptionUtil~SubscribeCallback} [params.subscribeCallback]
   * @returns {Promise} promise to be resolved when the whole component
   * tree is subscribed
   */
  function subscribeAll(_ref2) {
    var comps = _ref2.comps,
      sub = _ref2.sub,
      depth = _ref2.depth,
      subscriptionFilter = _ref2.subscriptionFilter,
      originalDepth = _ref2.originalDepth,
      subscribeCallback = _ref2.subscribeCallback;
    comps = comps.filter(isReadableComponent);
    if (comps.length === 0 || depth === 0 && !subscriptionFilter) {
      return Promise.resolve();
    }
    return Promise.resolve(sub.subscribe(comps)).then(function () {
      if (subscribeCallback) {
        return Promise.all(comps.map(function (comp) {
          return subscribeCallback({
            comp: comp,
            sub: sub
          });
        }));
      }
    }).then(function () {
      return getAllKids({
        comps: comps,
        subscriptionFilter: subscriptionFilter,
        depth: depth,
        originalDepth: originalDepth
      });
    }).then(function (allKids) {
      return subscribeAll({
        comps: allKids,
        sub: sub,
        depth: depth - 1,
        subscriptionFilter: subscriptionFilter,
        originalDepth: originalDepth,
        subscribeCallback: subscribeCallback
      });
    });
  }

  /**
   * Unsubscribe the entire component tree to the given depth.
   *
   * @inner
   * @param {Object} params the Object Literal for the method's arguments.
   * @param {Array.<baja.Component>} params.comps components to unsubscribe
   * @param {baja.Subscriber} params.sub the subscriber to use
   * @param {Number} params.depth the depth to unsubscribe - 0 unsubscribes nothing, 1
   * unsubscribes only the given components, 2 unsubscribes components and their
   * kids, etc.
   * @returns {Promise} promise to be resolved when the whole component
   * tree is unsubscribed
   */
  function unsubscribeAll(_ref3) {
    var comps = _ref3.comps,
      sub = _ref3.sub,
      depth = _ref3.depth;
    if (comps.length === 0 || depth === 0) {
      return Promise.resolve();
    }
    return getAllKids({
      comps: comps
    }).then(function (allKids) {
      return unsubscribeAll({
        comps: allKids,
        sub: sub,
        depth: depth - 1
      });
    }).then(function () {
      return sub.unsubscribe(comps);
    });
  }

  ////////////////////////////////////////////////////////////////
  // DepthSubscriber
  ////////////////////////////////////////////////////////////////

  /**
   * A wrapper for a `baja.Subscriber` that will subscribe/unsubscribe an entire
   * component tree.
   *
   * @param {Object} params The depth or the object literal for the parameters.
   * @param {Number} [params.depth] the depth to which to subscribe the component tree
   * @param {module:nmodule/webEditors/rc/wb/util/subscriptionUtil~SubscriptionFilter} [params.subscriptionFilter]
   * Starting in Niagara 4.13, if the optional subscriptionFilter function is provided, it will be called for each
   * potentially subscribable BComponent with its current depth. If given, the `depth` parameter
   * will be _ignored_: the DepthSubscriber will crawl the whole component tree and subscribe
   * everything that matches the filter. Returning false from the filter will not subscribe that
   * component, and its children will not be checked. Please note that this only applies to
   * descendent components; components passed directly to `subscribe()` will always be subscribed
   * even if they do not match `subscriptionFilter`.
   * @param {module:nmodule/webEditors/rc/wb/util/subscriptionUtil~SubscribeCallback} [params.subscribeCallback]
   * Starting in Niagara 4.13, if the optional subscribeCallback function is provided, it will receive a callback when a component is subscribed.
   * This can allow you to add additional subscriptions outside the normal depth and filter results.
   * @throws {Error} if neither a depth nor a subscriptionFilter is provided.
   *
   * @alias module:nmodule/webEditors/rc/fe/baja/util/DepthSubscriber
   * @class
   */
  var DepthSubscriber = function DepthSubscriber(params) {
    var that = this;
    var depth = -1;
    var subscriptionFilter;
    var subscribeCallback;
    that.$params = params;
    if (typeof params === 'number') {
      depth = params;
    } else if (params) {
      depth = params.depth || (params.subscriptionFilter ? 0 : -1);
      subscriptionFilter = that.$subscriptionFilter = params.subscriptionFilter;
      subscribeCallback = that.$subscribeCallback = params.subscribeCallback;
    }
    if (!subscriptionFilter && (typeof depth !== 'number' || depth < 0)) {
      throw new Error('depth must be 0 or greater');
    }
    var sub = new baja.Subscriber();
    that.$depth = depth;
    that.$subscriber = sub;
    that.$directComps = [];
    sub.attach('added', function (prop) {
      var comp = this.get(prop);
      var depth = comp.getType().is('baja:Component') ? that.getDepth(comp) : -1;
      if (depth >= 0) {
        if (subscriptionFilter && !subscriptionFilter({
          component: comp,
          depth: depth
        })) {
          return;
        }
        subscribeAll({
          comps: [comp],
          sub: sub,
          depth: that.$depth - depth,
          subscriptionFilter: subscriptionFilter,
          originalDepth: that.$depth,
          subscribeCallback: subscribeCallback
        })["catch"](logError);
      }
    });
  };

  /**
   * This will return a new Depth Subscriber with the same parameters.
   * @param [params] If any parameters are provided, they will override the parameters from the original subscriber.
   * @return {module:nmodule/webEditors/rc/fe/baja/util/DepthSubscriber | DepthSubscriber}
   * @since Niagara 4.13
   */
  DepthSubscriber.prototype.clone = function (params) {
    if (typeof params === 'number') {
      params = {
        depth: params
      };
    }
    return new DepthSubscriber(Object.assign({}, this.$params, params));
  };

  /**
   * Returns an array of the top level components currently subscribed
   * by this 'DepthSubscriber'
   * @returns {Array.<baja.Component>} a copy of the array of
   * components subscribed by this 'DepthSubscriber'
   */
  DepthSubscriber.prototype.getComponents = function () {
    return this.$directComps.slice();
  };

  /**
   * Returns an array of any components currently subscribed
   * by this 'DepthSubscriber'
   * @returns {Array.<baja.Component>} a copy of the array of
   * components subscribed by this 'DepthSubscriber'
   * @since Niagara 4.13
   */
  DepthSubscriber.prototype.getAllComponents = function () {
    return this.$subscriber.getComponents();
  };

  /**
   * Return true if the 'DepthSubscriber' is empty of subscriptions
   * @returns {Boolean} true if the 'DepthSubscriber' is empty
   */
  DepthSubscriber.prototype.isEmpty = function () {
    return this.$directComps.length === 0;
  };

  /**
   * This will return true when there is a subscriptionFilter.
   * @return {boolean}
   * @since Niagara 4.13
   */
  DepthSubscriber.prototype.hasSubscriptionFilter = function () {
    return !!this.$subscriptionFilter;
  };

  /**
   * Get the depth of the given component. If the component given is directly
   * subscribed by this `DepthSubscriber`, the result will be 0, a child
   * component will give 1, etc.
   *
   * If no component parameter given, returns the configured depth (`depth`
   * param to constructor).
   *
   * @param {baja.Complex} [comp]
   * @returns {Number} depth of the given component, or -1 if component is
   * not subscribed by this subscriber. Note that even if the component is
   * a descendant of a component subscribed by this `DepthSubscriber`, if it is
   * outside the configured subscribe depth, -1 will still be returned unless
   * the DepthSubscriber has a subscriptionFilter function that returns true for that component.
   */
  DepthSubscriber.prototype.getDepth = function (comp) {
    var directComps = this.$directComps;
    var subscriptionFilter = this.$subscriptionFilter;
    var $depth = this.$depth;
    var originalComp = comp;
    if (!comp) {
      return $depth;
    }
    if (!baja.hasType(comp, 'baja:Complex')) {
      return -1;
    }
    var depth = 0;
    while (comp) {
      if (directComps.indexOf(comp) >= 0) {
        if (depth < $depth) {
          return depth;
        }
        if (subscriptionFilter && subscriptionFilter({
          component: originalComp,
          depth: depth
        })) {
          return depth;
        }
        return -1;
      }
      depth++;
      comp = comp.getParent();
    }
    return -1;
  };

  /**
   * Attaches events to the backing `Subscriber`.
   */
  DepthSubscriber.prototype.attach = function () {
    var sub = this.$subscriber;
    return sub.attach.apply(sub, arguments);
  };

  /**
   * Detaches events from the backing `Subscriber`.
   */
  DepthSubscriber.prototype.detach = function () {
    var sub = this.$subscriber;
    return sub.detach.apply(sub, arguments);
  };

  /**
   * Return true if the backing `Subscriber` has the given component subscribed.
   *
   * @param {baja.Component} comp
   * @returns {Boolean}
   */
  DepthSubscriber.prototype.isSubscribed = function (comp) {
    return this.$subscriber.isSubscribed(comp);
  };

  /**
   * Subscribes the given components to the depth specified in the constructor.
   * @param {baja.Component|Array.<baja.Component>|Object} comps
   * @returns {Promise} promise to be resolved when the component tree
   * is fully subscribed
   */
  DepthSubscriber.prototype.subscribe = function (comps) {
    var _this = this;
    comps = baja.objectify(comps, 'comps');
    var arr = toArray(comps.comps);
    return subscribeAll({
      comps: arr,
      sub: this.$subscriber,
      depth: this.$depth,
      subscriptionFilter: this.$subscriptionFilter,
      originalDepth: this.$depth,
      subscribeCallback: this.$subscribeCallback
    }).then(function () {
      _this.$directComps = union(_this.$directComps, arr);
    }).then(okHandler(comps.ok), failHandler(comps.fail));
  };

  /**
   * Unsubscribes the given components to the depth specified in the
   * constructor.
   * @param {baja.Component|Array.<baja.Component>|Object} comps
   * @returns {Promise} promise to be resolved when the component tree
   * is full unsubscribed
   */
  DepthSubscriber.prototype.unsubscribe = function (comps) {
    var _this2 = this;
    comps = baja.objectify(comps, 'comps');
    var arr = toArray(comps.comps);
    return unsubscribeAll({
      comps: arr,
      sub: this.$subscriber,
      depth: this.$depth,
      subscribeCallback: this.$subscribeCallback
    }).then(function () {
      _this2.$directComps = difference(_this2.$directComps, arr);
    }).then(okHandler(comps.ok), failHandler(comps.fail));
  };

  /**
   * Unsubscribes everything currently subscribed, regardless of depth.
   *
   * @returns {Promise}
   */
  DepthSubscriber.prototype.unsubscribeAll = function () {
    var _this3 = this;
    return this.$subscriber.unsubscribeAll().then(function () {
      _this3.$directComps = [];
    });
  };
  function okHandler(ok) {
    return function (result) {
      if (ok) {
        ok(result);
      }
      return result;
    };
  }
  function failHandler(fail) {
    return function (err) {
      if (fail) {
        fail(err);
      }
      throw err;
    };
  }
  return DepthSubscriber;
});
