/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/baja/util/spaceUtils', 'nmodule/webEditors/rc/fe/baja/util/subscriberUtils', 'nmodule/webEditors/rc/fe/baja/util/typeUtils'], function (baja, Promise, _, spaceUtils, subscriberUtils, typeUtils) {
  'use strict';

  var partial = _.partial;
  var resolveService = spaceUtils.resolveService;
  var leaseWhenNeeded = subscriberUtils.leaseWhenNeeded;
  var isComplex = typeUtils.isComplex;

  /**
   * API Status: **Private**
   * @exports nmodule/webEditors/rc/fe/baja/util/userUtils
   */
  var exports = {};
  exports.$getUserService = leaseWhenNeeded(partial(resolveService, 'baja:UserService'));

  /**
   * Resolve the `baja:User` currently logged in.
   *
   * @param {object|baja.Component} [params] resolution params, or can be the
   * `base` object directly
   * @param {baja.Component} [params.base] a base object to resolve the
   * `UserService` against
   * @param {boolean} [params.lease=true] if true, the `User` will be leased
   * before being resolved
   * @param {number} [params.leaseTime] time in millis to lease the user,
   * otherwise the default lease time will be used
   * @returns {Promise.<baja.Component>} to be resolved with the `baja:User`
   * currently logged in
   */
  exports.resolveCurrentUser = function () {
    var params = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    if (isComplex(params)) {
      params = {
        base: params
      };
    }
    var _params = params,
      base = _params.base,
      _params$lease = _params.lease,
      lease = _params$lease === void 0 ? true : _params$lease,
      leaseTime = _params.leaseTime;
    return exports.$getUserService(base).then(function (userService) {
      var user = userService.get(baja.SlotPath.escape(baja.getUserName()));
      if (lease) {
        return user.lease({
          time: leaseTime
        }).then(function () {
          return user;
        });
      } else {
        return user;
      }
    });
  };

  /**
   * Determines if the current user is a super user or not.
   * If you are attempting to support an offline environment you may want to call `isSuperUserOrOffline()` instead to avoid
   * problems with attempting to resolve a user in an offline environment.
   *
   * @param {object|baja.Component} [params] resolution params, or can be the
   * `base` object directly.
   * @param {baja.Component} [params.base] a base object to resolve the
   * `UserService` against.
   * @param {boolean} [params.lease=true] if true, the `User` will be leased
   * before being resolved.
   * @param {number} [params.leaseTime] time in millis to lease the user,
   * otherwise the default lease time will be used.
   * @returns {Promise.<boolean>}
   */
  exports.isSuperUser = function (params) {
    return exports.resolveCurrentUser(params).then(function (user) {
      return user.get("permissions").isSuperUser();
    });
  };

  /**
   * Determines if the current user is a super user or not and if this is an offline
   * environment, then there is no user to limit permissions and this function returns true.
   *
   * @param {object|baja.Component} [params] resolution params, or can be the
   * `base` object directly.
   * @param {baja.Component} [params.base] a base object to resolve the
   * `UserService` against.
   * @param {boolean} [params.lease=true] if true, the `User` will be leased
   * before being resolved.
   * @param {number} [params.leaseTime] time in millis to lease the user,
   * otherwise the default lease time will be used.
   * @returns {Promise.<boolean>}
   */
  exports.isSuperUserOrOffline = function (params) {
    return Promise.resolve(baja.isOffline() || exports.isSuperUser(params));
  };
  return exports;
});
