/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/*eslint-env browser */ /*jshint browser: true */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/audio/AudioManager
 */
define(["jquery", "underscore"], function ($, _) {
  'use strict';

  /**
   * The base class for all audio clips.
   *
   * @private
   * @class
   * @alias module:nmodule/webEditors/rc/wb/audio/AudioManager~AudioClip
   * @param {baja.file.File} audioFile The audio file for the clip.
   */
  var AudioClip = function AudioClip(audioFile) {
    this.$audioFile = audioFile;
  };

  /**
   * Plays an audio clip.
   */
  AudioClip.prototype.play = function () {};

  /**
   * Stops an audio clip.
   */
  AudioClip.prototype.stop = function () {};

  /**
   * An HTML5 audio implementation.
   *
   * @private
   * @class
   * @alias module:nmodule/webEditors/rc/wb/audio/AudioManager~Html5Audio
   * @extends module:nmodule/webEditors/rc/wb/audio/AudioManager~AudioClip
   * @param {baja.file.File} audioFile The audio file for the clip.
   * @param {Audio} [audio]
   */
  var Html5Audio = function Html5Audio(audioFile, audio) {
    AudioClip.apply(this, arguments);
    this.$audio = audio || new Audio(audioFile.getReadUri());
  };
  Html5Audio.prototype = Object.create(AudioClip.prototype);
  Html5Audio.prototype.constructor = Html5Audio;
  Html5Audio.prototype.play = function () {
    this.$audio.play();
  };
  Html5Audio.prototype.stop = function () {
    this.$audio.pause();
    this.$audio.currentTime = 0;
  };

  /**
   * A Internet Explorer audio implementation for WAV files.
   *
   * This class isn't pretty but it was the only way I could effectively play a WAV file
   * using Internet Explorer.
   *
   * @private
   * @class
   * @alias module:nmodule/webEditors/rc/wb/audio/AudioManager~BgSoundAudio
   * @extends module:nmodule/webEditors/rc/wb/audio/AudioManager~AudioClip
   * @param {baja.file.File} audioFile The audio file for the clip.
   * @param {jQuery} [audio]
   */
  var BgSoundAudio = function BgSoundAudio(audioFile, audio) {
    var that = this;
    AudioClip.apply(that, arguments);

    // Lazily create the background sound element we're going to use.
    that.$audio = audio || $("#niagaraAudioManagerBgSnd");
    if (!that.$audio.get(0)) {
      that.$audio = $("<bgsound id='niagaraAudioManagerBgSnd'/>").append("body");
    }
  };
  BgSoundAudio.prototype = Object.create(AudioClip.prototype);
  BgSoundAudio.prototype.constructor = BgSoundAudio;

  /**
   * @returns {Boolean} Returns true if the detected environment is supported.
   */
  BgSoundAudio.isSupported = function (wnd) {
    wnd = wnd || window;
    return !!(wnd && wnd.navigator && wnd.navigator.userAgent && (wnd.navigator.userAgent.indexOf("MSIE") > -1 || wnd.navigator.userAgent.indexOf("Trident") > -1));
  };
  BgSoundAudio.prototype.play = function () {
    // Setting these elements will replay the sound.
    var that = this;
    that.$audio.attr("src", that.$audioFile.getReadUri());
    that.$audio.attr("volume", 0);
  };
  BgSoundAudio.prototype.stop = function () {
    // For the antiquated 'bgsound' element there appears to be no stop so instead just turn
    // the volume down.
    this.$audio.attr("volume", -10000);
  };

  /**
   * Plays sound files in a Workbench environment.
   *
   * @private
   * @class
   * @alias module:nmodule/webEditors/rc/wb/audio/AudioManager~WorkbenchAudio
   * @extends module:nmodule/webEditors/rc/wb/audio/AudioManager~AudioClip
   * @param {baja.file.File} audioFile The audio file for the clip.
   */
  var WorkbenchAudio = function WorkbenchAudio(audioFile) {
    AudioClip.apply(this, arguments);
    this.$ordStr = String(this.$audioFile.getOrdInSession());
  };
  WorkbenchAudio.prototype = Object.create(AudioClip.prototype);
  WorkbenchAudio.prototype.constructor = WorkbenchAudio;

  /**
   * @returns {Boolean} Returns true if the detected environment is supported.
   */
  WorkbenchAudio.isSupported = function (wnd) {
    wnd = wnd || window;
    return !!(wnd && wnd.niagara && wnd.niagara.wb && wnd.niagara.wb.util && typeof wnd.niagara.wb.util.playSound === "function" && typeof wnd.niagara.wb.util.stopSound === "function");
  };
  WorkbenchAudio.prototype.play = function (wnd) {
    (wnd || window).niagara.wb.util.playSound(this.$ordStr);
  };
  WorkbenchAudio.prototype.stop = function (wnd) {
    (wnd || window).niagara.wb.util.stopSound(this.$ordStr);
  };

  /**
   * Manages the playing of audio player clips. Please use the singleton
   * `INSTANCE` when using the audio manager.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/audio/AudioManager
   */
  var AudioManager = function AudioManager() {
    this.$clips = {};
  };

  /**
   * The primary instance for of the `AudioManager`.
   *
   * @type {AudioManager}
   */
  AudioManager.INSTANCE = new AudioManager();

  /**
   * The internal HTML5 browser audio constructor.
   * This is exposed for testing purposes only.
   *
   * @private
   * @internal
   *
   * @type {module:nmodule/webEditors/rc/wb/audio/AudioManager~Html5Audio}
   */
  AudioManager.Html5Audio = Html5Audio;

  /**
   * The bgsound audio constructor. Used for IE WAV files only.
   * This is exposed for testing purposes only.
   *
   * @private
   * @internal
   *
   * @type {module:nmodule/webEditors/rc/wb/audio/AudioManager~BgSoundAudio}
   */
  AudioManager.BgSoundAudio = BgSoundAudio;

  /**
   * The workbench audio constructor.
   * This is exposed for testing purposes only.
   *
   * @private
   * @internal
   *
   * @type {module:nmodule/webEditors/rc/wb/audio/AudioManager~WorkbenchAudio}
   */
  AudioManager.WorkbenchAudio = WorkbenchAudio;

  /**
   * Make an audio object and return it.
   *
   * @internal
   *
   * @param  {baja.file.File} audioFile The audio file to play.
   * @returns {module:nmodule/webEditors/rc/wb/audio/AudioManager~AudioClip} An audio object.
   */
  AudioManager.prototype.$makeAudio = function (audioFile) {
    if (WorkbenchAudio.isSupported()) {
      return new WorkbenchAudio(audioFile);
    } else if (BgSoundAudio.isSupported() && audioFile.getMimeType() === "audio/wav") {
      return new BgSoundAudio(audioFile);
    } else {
      return new Html5Audio(audioFile);
    }
  };

  /**
   * Play the audio file.
   *
   * @param {baja.file.File} audioFile The audio file to play.
   */
  AudioManager.prototype.play = function (audioFile) {
    var that = this,
      uri = audioFile.getReadUri(),
      clip = that.$clips[uri];
    if (!clip) {
      that.$clips[uri] = clip = that.$makeAudio(audioFile);
    }
    clip.play();
  };

  /**
   * Stop the audio file from currently playing.
   *
   * @param  {baja.file.File} audioFile The audio file to stop playing.
   */
  AudioManager.prototype.stop = function (audioFile) {
    var clip = this.$clips[audioFile.getReadUri()];
    if (clip) {
      clip.stop();
    }
  };

  /**
   * Stop all audio clips playing.
   */
  AudioManager.prototype.stopAll = function () {
    _.each(this.$clips, function (clip) {
      clip.stop();
    });
  };
  return AudioManager;
});
