/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/commands/multisheet/ToggleSlotModeCommand
 */
define(['baja!', 'bajaux/commands/ToggleCommand', 'bajaux/events'], function (baja, ToggleCommand, events) {
  'use strict';

  var MODIFY_EVENT = events.MODIFY_EVENT,
    UNMODIFY_EVENT = events.UNMODIFY_EVENT;

  /**
   * Check whether I have admin write permissions on the given component.
   *
   * @inner
   * @param {baja.Component} comp
   * @returns {Boolean}
   */
  function hasAdminWrite(comp) {
    return baja.hasType(comp, 'baja:Component') && comp.getPermissions().hasAdminWrite();
  }

  /**
   * Check whether the multisheet is in a state where I should be allowed to
   * view slot structure details.
   *
   * @inner
   * @param {module:nmodule/webEditors/rc/wb/MultiSheet} multiSheet
   * @returns {Boolean}
   */
  function canToggleSlotMode(multiSheet) {
    return !multiSheet.isModified() && hasAdminWrite(multiSheet.value());
  }

  /**
   * Command for toggling whether a MultiSheet is showing slot structure
   * details or field editors.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/commands/multisheet/ToggleSlotModeCommand
   * @extends module:bajaux/commands/ToggleCommand
   * @param {module:nmodule/webEditors/rc/wb/MultiSheet} multiSheet
   */
  var ToggleSlotModeCommand = function ToggleSlotModeCommand(multiSheet) {
    var that = this;
    ToggleCommand.call(that, {
      module: 'webEditors',
      lex: 'commands.toggleSlotMode',
      selected: multiSheet.isSlotMode(),
      enabled: false,
      func: function func() {
        that.toggle();
        //TODO: revisit - i can't change my own selected status
        return multiSheet.setSlotMode(that.isSelected());
      }
    });
    function updateEnabled() {
      that.setEnabled(canToggleSlotMode(multiSheet));
    }
    multiSheet.on('loaded', function () {
      updateEnabled();
      multiSheet.jq().on(MODIFY_EVENT + ' ' + UNMODIFY_EVENT, updateEnabled);
    });
  };
  ToggleSlotModeCommand.prototype = Object.create(ToggleCommand.prototype);
  ToggleSlotModeCommand.prototype.constructor = ToggleSlotModeCommand;
  return ToggleSlotModeCommand;
});
