function _slicedToArray(r, e) { return _arrayWithHoles(r) || _iterableToArrayLimit(r, e) || _unsupportedIterableToArray(r, e) || _nonIterableRest(); }
function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
function _iterableToArrayLimit(r, l) { var t = null == r ? null : "undefined" != typeof Symbol && r[Symbol.iterator] || r["@@iterator"]; if (null != t) { var e, n, i, u, a = [], f = !0, o = !1; try { if (i = (t = t.call(r)).next, 0 === l) { if (Object(t) !== t) return; f = !1; } else for (; !(f = (e = i.call(t)).done) && (a.push(e.value), a.length !== l); f = !0); } catch (r) { o = !0, n = r; } finally { try { if (!f && null != t["return"] && (u = t["return"](), Object(u) !== u)) return; } finally { if (o) throw n; } } return a; } }
function _arrayWithHoles(r) { if (Array.isArray(r)) return r; }
/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/*eslint-env browser */ /*jshint browser: true */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/job/JobLogWidget
 */
define(['baja!', 'baja!baja:Job', 'hbs!nmodule/webEditors/rc/wb/job/template/JobLogWidget', 'lex!webEditors', 'log!nmodule.webEditors.rc.wb.job.JobLogWidget', 'dialogs', 'jquery', 'Promise', 'underscore', 'bajaux/Widget', 'bajaux/commands/Command', 'bajaux/commands/CommandGroup', 'bajaux/mixin/subscriberMixIn', 'bajaux/util/CommandButtonGroup', 'nmodule/webEditors/rc/fe/BaseWidget', 'nmodule/webEditors/rc/wb/job/JobBar', 'nmodule/webEditors/rc/wb/job/jobSupport', 'nmodule/webEditors/rc/wb/table/Table'], function (baja, types, tplJobLogWidget, lexs, log, dialogs, $, Promise, _, Widget, Command, CommandGroup, subscribable, CommandButtonGroup, BaseWidget, JobBar, jobSupport, Table) {
  "use strict";

  var makeRow = jobSupport.makeLogTableRow,
    makeModel = jobSupport.makeLogModel,
    showDetails = jobSupport.showItemDetailsDialog,
    webEditorsLex = lexs[0],
    logError = log.severe.bind(log),
    scrollToLastRow,
    TABLE_SCROLL_DEBOUNCE_MS = 250,
    LOG_POLL_TIMER_MS = 3000,
    CANCEL_BUTTON_INDEX = 0;
  var CELL_ACTIVATED_EVENT = Table.CELL_ACTIVATED_EVENT;
  function hasJobType(obj) {
    return baja.hasType(obj, 'baja:Job');
  }

  ////////////////////////////////////////////////////////////////
  // DOM Utility Functions
  ////////////////////////////////////////////////////////////////

  function getTableElement(jq) {
    return jq.find('.job-log-table');
  }
  function getTableModel(tableElem) {
    return Widget["in"](tableElem).getModel();
  }
  function getJobWidgetElement(jq) {
    return jq.find('.JobLogWidget');
  }

  ////////////////////////////////////////////////////////////////
  // Cancel Command
  ////////////////////////////////////////////////////////////////

  /**
   * Command used to cancel a running job.
   *
   * @private
   * @class
   * @extends module:bajaux/commands/Command
   * @param {baja.Component} job - the job instance
   */
  var CancelCommand = function CancelCommand(job) {
    Command.call(this, {
      displayName: webEditorsLex.get('JobDialog.cancel'),
      icon: 'module://icons/x16/stop.png',
      enabled: true,
      /**
       * Get a confirmation from the user, then call the cancel action.
       *
       * @returns {Promise} the `Promise` of the 'yes/no' dialog.
       */
      func: function func() {
        return dialogs.showYesNo(webEditorsLex.get('JobDialog.confirmCancel')).promise().then(function (_ref) {
          var _ref2 = _slicedToArray(_ref, 2),
            dlg = _ref2[0],
            buttonClicked = _ref2[1];
          if (buttonClicked === 'yes') {
            return job.cancel();
          }
        });
      }
    });
  };
  CancelCommand.prototype = Object.create(Command.prototype);
  CancelCommand.prototype.constructor = CancelCommand;

  ////////////////////////////////////////////////////////////////
  // Close Command
  ////////////////////////////////////////////////////////////////

  /**
   * Command used for the close button on the dialog. This will destroy
   * the widgets, then close the dialog.
   *
   * @private
   * @class
   * @extends module:bajaux/commands/Command
   * @param {module:dialogs~Dialog} dlg the dialog instance
   */
  var CloseCommand = function CloseCommand(dlg) {
    var that = this;
    Command.call(that, {
      displayName: webEditorsLex.get('JobDialog.close'),
      icon: 'module://icons/x16/close.png',
      enabled: true,
      /**
       * Destroy the widgets, then close the dialog.
       * @returns {Promise}
       */
      func: function func() {
        var jq = dlg.jq(),
          jobWidget = Widget["in"](getJobWidgetElement(jq)),
          commandButtons = Widget["in"](getCommandButtonGroupElement(jq)),
          promises = [jobWidget.destroy(), commandButtons.destroy()];
        return Promise.all(promises).then(function () {
          dlg.close();
        });
      }
    });
  };
  CloseCommand.prototype = Object.create(Command.prototype);
  CloseCommand.prototype.constructor = CancelCommand;

  ////////////////////////////////////////////////////////////////
  // Job Event Handling
  ////////////////////////////////////////////////////////////////

  function scrollRowIntoView(row) {
    if (row.scrollIntoView && typeof row.scrollIntoView === 'function') {
      row.scrollIntoView(false);
    }
  }

  /**
   * De-bounced function to scroll to the bottom table row when new items
   * are inserted following a log update. We want to let updates to the
   * table settle before we attempt the scroll.
   */
  scrollToLastRow = _.debounce(function (widget, tableElem) {
    var rows, last;
    if (widget.isDestroyed()) {
      return;
    }
    rows = tableElem.find('tr');
    if (rows.length) {
      last = rows.last().get(0);
      scrollRowIntoView(last);
    }
  }, TABLE_SCROLL_DEBOUNCE_MS);

  /**
   * Function called when the job's log has been updated with a new item
   * and the dialog's table should be updated with the new row(s).
   *
   * @param widget - the job log widget.
   * @param {Array.<Object>} items - the job log items parsed from the event.
   * @param {Number} startIndex - the index at which the new items should be inserted in the
   * table model.
   *
   * @returns {Promise}
   */
  function insertLogItems(widget, items, startIndex) {
    if (widget.isDestroyed()) {
      return Promise.resolve();
    }
    var tableElem = getTableElement(widget.jq()),
      model = getTableModel(tableElem),
      rows = _.map(items, makeRow);
    return model.insertRows(rows, startIndex).then(function () {
      scrollToLastRow(widget, tableElem);
    });
  }

  /**
   * Test the end time property of the job to determine whether it is complete.
   */
  function isJobFinished(job) {
    return !job.get('endTime').equals(baja.AbsTime.DEFAULT);
  }

  /**
   * Function called to update the dialog's UI when the job completes.
   *
   * @param {baja.Component} job - the job
   * @param widget - the job log widget instance
   */
  function jobFinished(job, widget) {
    var success, jq;
    if (widget.isDestroyed()) {
      return;
    }
    jq = widget.jq();
    success = job.getJobState().getTag() === 'success';
    jq.find('.status-container').addClass('status-' + (success ? 'success' : 'fail'));
  }

  /**
   * Configure handlers for the events on the job component. We are interested
   * in a few things - the 'endTime' property, to know when the job is finished,
   * and the 'progress' property, to give a hint when we should try polling for new
   * log items. This is also where we will configure and manage the poll timer.
   *
   * @param {baja.Component} job - the job
   * @param widget - the job log widget instance
   */
  function configureJobUpdates(job, widget) {
    var jq = widget.jq(),
      readLatest,
      timeoutId,
      PROGRESS_UPDATE_THROTTLE_MS = 1000,
      finished = false;

    /**
     * Set a timer to poll for any new entries in the log.
     */
    function setLogPollTimer(job) {
      timeoutId = window.setTimeout(function () {
        timeoutId = undefined;
        readLatestEntries(job).then(function () {
          if (!finished) {
            setLogPollTimer(job);
          }
        })["catch"](logError);
      }, LOG_POLL_TIMER_MS);
    }

    /**
     * Clear the timer for polling the job log. This will be called
     * either due to job completion, or a progress event coming in
     * before the current timer has fired.
     */
    function clearLogPollTimer() {
      if (timeoutId) {
        window.clearTimeout(timeoutId);
        timeoutId = undefined;
      }
    }

    /**
     * Remove all the items from the table in response to the JobLog
     * being reset.
     */
    function resetTableModel() {
      var tableElem = getTableElement(jq),
        model = getTableModel(tableElem),
        count = model.getRows().length;
      return count ? model.removeRows(0, count) : Promise.resolve();
    }

    /**
     * Read the latest entries from the log, beginning after the
     * last sequence number we received. Note that if the job is
     * generating log entries faster than we can read them and
     * the size of the log is capped, the sequence number of the
     * next item we read may be greater than the current sequence
     * number + 1.
     */
    function readLatestEntries(job) {
      var initialSeq, lastSeq, logSequence;
      return job.readLogFrom(widget.$lastSeq + 1).then(function (seq) {
        logSequence = seq;
        initialSeq = logSequence.getInitialSequenceNumber();
        lastSeq = logSequence.getLastSequenceNumber();
        if (initialSeq > widget.$initialSeq) {
          widget.$initialSeq = initialSeq;
          return resetTableModel();
        }
      }).then(function () {
        if (lastSeq > widget.$lastSeq) {
          widget.$lastSeq = lastSeq;
          return jobSupport.parseEncodedLog(logSequence.getEncodedItems()).then(function (items) {
            return insertLogItems(widget, items);
          });
        }
      });
    }

    // Create a throttled version of the readLatestEntries() function to call
    // after an update on the progress property.

    readLatest = _.throttle(readLatestEntries, PROGRESS_UPDATE_THROTTLE_MS);

    /**
     * Look for an indication of progress or that the job is finished.
     * If it's finished, we can unhook the event handlers and try to
     * get any final items in the log. If the progress has changed, stop
     * the pending timer, try to get the latest changes, then start the timer
     * again.
     */
    function propChanged(prop) {
      if (prop.getName() === 'endTime' && isJobFinished(job)) {
        finished = true;
        clearLogPollTimer();
        detachJobEvents(job, widget);
        readLatestEntries(job)["catch"](logError);
        jobFinished(job, widget);
      } else if (prop.getName() === 'progress') {
        if (!finished) {
          clearLogPollTimer();
          readLatest(job).then(function () {
            if (!finished) {
              setLogPollTimer(job);
            }
          })["catch"](logError);
        }
      }
      return null;
    }
    widget.$handlers = {
      'changed': propChanged
    };
    job.attach(widget.$handlers);

    // Start the timer, to check for any new items if we don't receive any progress
    // events for a while (or at all).

    setLogPollTimer(job);
  }

  /**
   * Detach the event handler functions added by `configureJobUpdates`. Called when
   * the job completes or the widget is destroyed, whichever comes first.
   *
   * @param {baja.Component} job - the job
   * @param widget - the job log widget instance
   */
  function detachJobEvents(job, widget) {
    if (widget.$handlers) {
      job.detach(widget.$handlers);
      delete widget.$handlers;
    }
  }

  /**
   * Widget providing the ability to pop up a dialog showing the live progress of
   * a running BJob. This dialog contains a progress bar, with a table containing
   * the items from the job's log. The table will update dynamically by periodically
   * reading the latest items.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/job/JobLogWidget
   * @extends module:nmodule/webEditors/rc/fe/BaseWidget
   *
   * @param {Object} [params] - optional parameter object.
   * @param {Number} [params.truncateLength] - parameter to optionally truncate the text in the
   * main table to a maximum length.
   * @param {Boolean} [params.hideJobBarButtons] - boolean flag indicating whether to hide
   * the buttons on the job bar. Used when the widget is shown in a dialog.
   */
  var JobLogWidget = function JobLogWidget(params) {
    var that = this;
    params = params || {};
    that.$truncateLength = params.truncateLength;
    that.$hideJobBarButtons = !!params.hideJobBarButtons;
    BaseWidget.apply(that, arguments);
    subscribable(that);
  };
  JobLogWidget.prototype = Object.create(BaseWidget.prototype);
  JobLogWidget.prototype.constructor = JobLogWidget;

  /**
   * Initialize the widget. Creates a JobBar and Table as child widgets.
   *
   * @param {JQuery} dom
   * @returns {Promise}
   */
  JobLogWidget.prototype.doInitialize = function (dom) {
    var that = this;
    dom.addClass('JobLogWidget');
    that.$jobBar = new JobBar({
      hideCommandButtons: true
    });
    that.$table = new Table();
    function initializeJobBar() {
      return that.$jobBar.initialize($('<div>').appendTo(dom.find('.bar-container')));
    }
    function initializeLogTable() {
      var tableElem = $('<table class="ux-table job-log-table">');
      return that.$table.initialize(tableElem.appendTo(dom.find('.table-scroll-container')));
    }
    dom.html(tplJobLogWidget());
    return Promise.all([initializeJobBar(), initializeLogTable()]);
  };

  /**
   * Load the job into the widget. This will read the initial state of the job log,
   * load it into the child table, then configure the events to update the table.
   *
   * @param {baja.Component} value - the BJob component to be loaded into the widget.
   *
   * @returns {Promise}
   */
  JobLogWidget.prototype.doLoad = function (value) {
    var that = this;
    if (!baja.hasType(value, 'baja:Job')) {
      return Promise.reject(new Error('JobBar must be loaded with a BJob instance.'));
    }
    return that.$jobBar.load(value).then(function () {
      return value.readLogFrom(0);
    }).then(function (log) {
      that.$initialSeq = log.getInitialSequenceNumber();
      that.$lastSeq = log.getLastSequenceNumber();
      return jobSupport.parseEncodedLog(log.getEncodedItems());
    }).then(function (items) {
      var rows = _.map(items, makeRow),
        model = makeModel(rows, {
          sortable: false,
          truncateLength: that.$truncateLength
        });
      return that.$table.load(model);
    }).then(function () {
      configureJobUpdates(that.value(), that);
      that.$table.jq().on(CELL_ACTIVATED_EVENT, function (e, table, row) {
        showDetails(row.getSubject(), that.jq());
      });
    });
  };

  /**
   * Overrides doLayout() to adjust the height of the log table.
   */
  JobLogWidget.prototype.doLayout = function () {
    this.$layoutTable();
  };

  /**
   * Function called to adjust the height of the job log table. This will be
   * called in response to a widget layout request, and also when initially displayed
   * in a dialog.
   * @private
   */
  JobLogWidget.prototype.$layoutTable = function () {
    var jq = this.jq(),
      contentHeight = jq.height(),
      statusContainerHeight = jq.find('.status-container').outerHeight(true),
      tableContainerHeight;
    tableContainerHeight = contentHeight - statusContainerHeight;
    jq.find('.log-container').css({
      height: tableContainerHeight + 'px'
    });
  };

  /**
   * Destroy the widget and its children.
   *
   * @returns {Promise}
   */
  JobLogWidget.prototype.doDestroy = function () {
    var that = this;
    detachJobEvents(that.value(), that);
    return Promise.all([that.$jobBar.destroy(), that.$table.destroy()]).then(function () {
      that.jq().removeClass('JobLogWidget');
    });
  };

  ////////////////////////////////////////////////////////////////
  // Job Log Dialog
  ////////////////////////////////////////////////////////////////

  function getCommandButtonGroupElement(jq) {
    return jq.find('.job-dlg-command-container > .CommandButtonGroup');
  }
  function toggleButtonEnabled(dlg, index, enable) {
    var buttonGroupElement = getCommandButtonGroupElement(dlg.jq()),
      buttonGroup = Widget["in"](buttonGroupElement).value();
    buttonGroup.get(index).setEnabled(enable);
  }
  function toggleCancelEnabled(dlg, enable) {
    toggleButtonEnabled(dlg, CANCEL_BUTTON_INDEX, enable);
  }

  /**
   * Function to show the dialog once we have a reference to the job component and
   * have read in the initial contents of its log. This will create widgets for
   * the job bar, main log table, and command buttons. The job bar will be configured
   * to hide its buttons, but we will also create our own buttons in the content
   * area of the dialog. We choose this over using the dialog's own buttons as we want
   * the cancel button to perform an action, but not to cause the dialog to be dismissed.
   *
   * @param job - the job instance
   * @param params - the parameter object passed to the exported showFor() function.
   *
   * @returns {Promise}
   */
  function showDialog(job, params) {
    var logWidget = new JobLogWidget({
        truncateLength: params.truncateLength
      }),
      commandButtons = new CommandButtonGroup(),
      commandGroup = new CommandGroup(),
      propChanged,
      dialogParams;
    function initializeButtons(jq) {
      return commandButtons.initialize(jq);
    }
    function initializeWidgets(dom) {
      return logWidget.initialize($(dom.get(0))).then(function () {
        return initializeButtons($('<div>').appendTo($(dom.get(1))));
      });
    }

    /**
     * Load the widgets with values. We create the commands for the command group
     * here, as we will have a reference to the dialog at the point this
     * function is called, and we need to pass it to the commands.
     */
    function loadWidgets(job, dlg) {
      commandGroup.add(new CancelCommand(job), new CloseCommand(dlg));
      return Promise.all([logWidget.load(job, {}), commandButtons.load(commandGroup)]);
    }

    /**
     * Configure the table once the widgets have been initialized and the dialog has been
     * shown. This will fix the height of the table's container to take the available
     * vertical space in the middle of the dialog's content area.
     */
    function configureWidgetHeight(dlg) {
      var jq = dlg.jq(),
        contentHeight = dlg.content().height(),
        commandContainerHeight = jq.find('.job-dlg-command-container').outerHeight(true);
      jq.find('.job-dlg-widget-container').height(contentHeight - commandContainerHeight);
      logWidget.doLayout();
    }

    /**
     * Function passed to the dialog's static show() method to build its content before the
     * dialog is shown. This extends the dialog's show() function to create the widgets,
     * load their values, and then perform some custom layout on the main table. It also
     * hooks up the event handlers on the job, once we know the UI is created and ready.
     */
    function makeDialogContent(dlg, jq) {
      var superShow = dlg.show,
        dom = $('<div class="job-dlg-widget-container" /><div class="job-dlg-command-container"/>');
      jq.css({
        width: params.width,
        height: params.height
      });

      /*
       * Patch the dialog's show method to relayout the table for the new height
       * once the dialog has been shown and we can know the correct dimensions.
       */
      dlg.show = function () {
        var that = this;
        superShow.apply(that, arguments);
        configureWidgetHeight(dlg);
        return that;
      };
      return initializeWidgets(dom).then(function () {
        jq.append(dom);
        return loadWidgets(job, dlg, params);
      }).then(function () {
        var finished = isJobFinished(job);
        toggleCancelEnabled(dlg, !finished);
        propChanged = function propChanged(prop) {
          if (prop.getName() === 'endTime' && isJobFinished(job)) {
            toggleCancelEnabled(dlg, false);
          }
        };
        job.attach('changed', propChanged);
      });
    }
    dialogParams = {
      content: makeDialogContent
    };
    if (params.parent) {
      dialogParams.parent = params.parent;
    }
    return dialogs.show(dialogParams).promise().then(function () {
      if (propChanged) {
        job.detach('changed', propChanged);
      }
    });
  }

  /**
   * Static function to show the job log widget within a dialog for a single job. The argument to
   * this function may be either a BJob component instance, or an ord (String or `baja.Ord`) that
   * will resolve to a job. The second case is intended to be a simple convenience for a common
   * case where an Action is invoked on a component to create and start the job, and the job's ord
   * is the return value of that Action.
   *
   * The dialog's table will be initially populated with the contents of the job log at the
   * point the function is called. The job will be subscribed, which will then pick up further
   * log updates and add them to the table.
   *
   * The dialog will present two buttons to the user: one to cancel the running job, and one
   * to close the dialog.
   *
   * @static
   *
   * @param {String|baja.Ord|baja.Component} value a job component or an ord referencing one
   *
   * @param {Object} params an object literal containing the parameters for the dialog.
   *
   * @param {String} params.width width of the dialog expressed as a string using css units:
   * '30em', for example.
   *
   * @param {String} params.height height of the dialog expressed as a string using css units:
   * '30em', for example.
   *
   * @param {JQuery} [params.parent] - optional parent DOM element for the dialog.
   *
   * @param {Number} [params.truncateLength] optional parameter to specify that long messages
   * should be truncated to a maximum number of characters in the main table. The full text
   * of the item can still be obtained by double clicking the row to get the 'details' dialog.
   * Text will not be truncated by default.
   *
   * @returns {Promise} a `Promise` that will be resolved once the dialog has been closed.
   */
  JobLogWidget.showDialog = function (value, params) {
    var resolveJob;
    params = params || {};
    if (!params.width || !params.height) {
      return Promise.reject(new Error('must specify width and height parameters'));
    }
    if (typeof value === 'string') {
      value = baja.Ord.make(value);
    }
    resolveJob = value instanceof baja.Ord ? value.get() : Promise.resolve(value);
    return resolveJob.then(function (resolved) {
      if (!hasJobType(resolved)) {
        throw new Error('JobDialog.showFor() requires a BJob.');
      }
      return showDialog(resolved, params);
    });
  };
  return JobLogWidget;
});
