function _slicedToArray(r, e) { return _arrayWithHoles(r) || _iterableToArrayLimit(r, e) || _unsupportedIterableToArray(r, e) || _nonIterableRest(); }
function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
function _iterableToArrayLimit(r, l) { var t = null == r ? null : "undefined" != typeof Symbol && r[Symbol.iterator] || r["@@iterator"]; if (null != t) { var e, n, i, u, a = [], f = !0, o = !1; try { if (i = (t = t.call(r)).next, 0 === l) { if (Object(t) !== t) return; f = !1; } else for (; !(f = (e = i.call(t)).done) && (a.push(e.value), a.length !== l); f = !0); } catch (r) { o = !0, n = r; } finally { try { if (!f && null != t["return"] && (u = t["return"](), Object(u) !== u)) return; } finally { if (o) throw n; } } return a; } }
function _arrayWithHoles(r) { if (Array.isArray(r)) return r; }
/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/links/LinkEditor
 */
define(['baja!', 'baja!baja:Link,baja:Vector', 'jquery', 'Promise', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/wb/links/OrdChooser', 'nmodule/webEditors/rc/wb/tree/BajaComplexTreeNode', 'nmodule/webEditors/rc/wb/tree/TreeNode'], function (baja, types, $, Promise, feDialogs, BaseEditor, OrdChooser, BajaComplexTreeNode, TreeNode) {
  'use strict';

  ////////////////////////////////////////////////////////////////
  // Utility functions
  ////////////////////////////////////////////////////////////////

  /**
   * Compute the source ORD to be loaded into the incoming ORD chooser.
   *
   * @inner
   * @param {baja.Complex} complex
   * @param {baja.Complex} link `baja:Link`
   * @returns {Promise} promise to be resolved with source `baja.Ord`
   */
  function getSourceOrd(complex, link) {
    return link.get('sourceOrd').resolve({
      base: baja.station
    }).then(function (target) {
      var sourceOrd = target.getComponent().getNavOrd(),
        sourceSlot = link.get('sourceSlotName');
      return baja.Ord.make(sourceOrd + '/' + sourceSlot);
    });
  }

  /**
   * Compute the target ORD to be loaded into the outgoing ORD chooser.
   *
   * @inner
   * @param {baja.Component} complex
   * @param {baja.Complex} link `baja:Link`
   * @returns {baja.Ord} the target `baja.Ord`
   */
  function getTargetOrd(complex, link) {
    return baja.Ord.make(complex.getNavOrd() + '/' + link.get('targetSlotName'));
  }

  /**
   * Check to see if value is an instanceof BComponent but not BVector.
   *
   * @inner
   * @param {baja.Value} value
   * @returns {boolean}
   */
  function isNonVectorComponent(value) {
    return baja.hasType(value, 'baja:Component') && !value.getType().is('baja:Vector');
  }

  /**
   * Return true if the two links have the same ORD and slot data.
   *
   * @inner
   * @param {baja.Complex} newLink `baja:Link`
   * @param {baja.Complex} oldLink `baja:Link`
   * @returns {Boolean}
   */
  function linkEquivalent(newLink, oldLink) {
    return newLink && oldLink && newLink.get('sourceOrd').equivalent(oldLink.get('sourceOrd')) && newLink.get('sourceSlotName') === oldLink.get('sourceSlotName') && newLink.get('targetSlotName') === oldLink.get('targetSlotName');
  }

  ////////////////////////////////////////////////////////////////
  // Define tree node hierarchy to use for ord choosers
  ////////////////////////////////////////////////////////////////

  /**
   * Tree node for link editor ord choosers. Should prevent selection of invalid
   * link targets.
   *
   * @inner
   * @class
   */
  var LinkNode = function LinkNode() {
    BajaComplexTreeNode.apply(this, arguments);
  };
  LinkNode.prototype = Object.create(BajaComplexTreeNode.prototype);
  LinkNode.prototype.constructor = LinkNode;

  /**
   * Slots of Component type that are not instances of BVector are not valid
   * link targets. c.f. `BLinkPad#isValidSlotRow`.
   * @returns {boolean}
   */
  LinkNode.prototype.isSelectable = function () {
    return !isNonVectorComponent(this.value());
  };

  ////////////////////////////////////////////////////////////////
  // LinkEditor definition
  ////////////////////////////////////////////////////////////////

  /**
   * Editor to allow the user to edit a `baja:Link`, or, to create a new Link,
   * two separate ORDs.
   *
   * Supports the following `bajaux` Properties:
   *
   * - `editIncoming` (default: `true`) set to `false` to prevent editing the
   *   incoming ORD
   * - `editOutgoing` (default: `true`) set to `false` to prevent editing the
   *   outgoing ORD,
   * - `useHandles` (default: `true`) set to `false` to have the created link
   *   use the full ORD, not just the handle
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/wb/links/LinkEditor
   * @param {Object} [params]
   */
  var LinkEditor = function LinkEditor(params) {
    BaseEditor.apply(this, arguments);
  };
  LinkEditor.prototype = Object.create(BaseEditor.prototype);
  LinkEditor.prototype.constructor = LinkEditor;

  /**
   * Edit a link, or create a new one. Shows a dialog prompting the user to
   * edit the incoming and outgoing ORDs, resolves a `baja:Link` instance and
   * saves it to the `Complex` being edited.
   *
   * @param {Object} params
   * @param {baja.Complex} params.complex the Complex that will receive the
   * `Link` instance]
   * @param {baja.Slot|String} [params.slot] the slot containing the existing
   * link. If given, this link will be overwritten with the user-entered link.
   * @param {baja.Ord} [params.sourceOrd] if creating a new link, provide the
   * source ORD to be initially shown to and edited by the user.
   * @param {baja.Ord} [params.targetOrd] if creating a new link, provide the
   * target ORD to be initially shown to and edited by the user.
   * @param {Object} [params.properties] `bajaux` Properties to be passed to
   * the displayed `LinkEditor`
   * @param {boolean} [params.readonly] true to show the link editor readonly
   * @returns {Promise} promise to be resolved once the user has
   * selected new ORDs for the link and the link has been saved (or discarded,
   * if the user clicked Cancel)
   */
  LinkEditor.editLink = function (params) {
    params = params || {};
    var complex = params.complex,
      slot = params.slot,
      sourceOrd = params.sourceOrd,
      targetOrd = params.targetOrd;
    if (!baja.hasType(complex, 'baja:Complex') || slot && complex.has(slot) && !complex.getSlot(slot).getType().is('baja:Link') || !slot && (!sourceOrd || !targetOrd)) {
      return Promise.reject(new Error('complex and baja:Link required'));
    }
    var oldLink = slot && complex.get(slot);
    return feDialogs.showFor({
      value: slot ? complex.get(slot) : baja.$('baja:Component', {
        sourceOrd: sourceOrd,
        targetOrd: targetOrd
      }),
      properties: params.properties,
      readonly: params.readonly,
      type: LinkEditor,
      formFactor: 'compact'
    }).then(function (newLink) {
      if (newLink && !linkEquivalent(newLink, oldLink)) {
        return complex.set({
          slot: slot,
          value: newLink
        });
      }
    });
  };

  ////////////////////////////////////////////////////////////////
  // Private functions
  ////////////////////////////////////////////////////////////////

  /**
   * Get the element containing the incoming ORD chooser.
   *
   * @private
   * @returns {jQuery}
   */
  LinkEditor.prototype.$getIncomingElement = function () {
    return this.jq().children('.incoming').children('.OrdChooser');
  };

  /**
   * Get the incoming `OrdChooser`.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/links/OrdChooser}
   */
  LinkEditor.prototype.$getIncomingOrdChooser = function () {
    return this.$getIncomingElement().data('widget');
  };

  /**
   * Get the element containing the outgoing ORD chooser.
   *
   * @private
   * @returns {jQuery}
   */
  LinkEditor.prototype.$getOutgoingElement = function () {
    return this.jq().children('.outgoing').children('.OrdChooser');
  };

  /**
   * Get the outgoing `OrdChooser`.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/links/OrdChooser}
   */
  LinkEditor.prototype.$getOutgoingOrdChooser = function () {
    return this.$getOutgoingElement().data('widget');
  };

  ////////////////////////////////////////////////////////////////
  // LinkEditor bajaux implementation
  ////////////////////////////////////////////////////////////////
  /**
   * Resolve the root tree node for link editor ord choosers.
   *
   * @inner
   * @returns {Promise} promise to be resolved with a root `TreeNode`
   */
  function resolveRootNode() {
    return baja.Ord.make('station:|slot:/').get().then(function (station) {
      //TODO: come up with a non stupid way of doing this
      return new TreeNode('local', 'local', [new LinkNode(station)]);
    });
  }

  /**
   * Initializes the incoming and outgoing `OrdChooser`s.
   *
   * @param {jQuery} dom
   * @returns {Promise} promise to be resolved when all ORD choosers
   * have been initialized
   */
  LinkEditor.prototype.doInitialize = function (dom) {
    var formFactor = this.getFormFactor(),
      props = this.properties(),
      readonly = this.isReadonly(),
      inReadonly = props.getValue('editIncoming') === false || readonly,
      outReadonly = props.getValue('editOutgoing') === false || readonly,
      inChooser = new OrdChooser({
        formFactor: formFactor,
        readonly: inReadonly
      }),
      outChooser = new OrdChooser({
        formFactor: formFactor,
        readonly: outReadonly
      }),
      inDiv = $('<div/>').appendTo($('<div class="incoming"/>').appendTo(dom)),
      outDiv = $('<div/>').appendTo($('<div class="outgoing"/>').appendTo(dom));
    inChooser.$resolveStationTreeNode = resolveRootNode;
    outChooser.$resolveStationTreeNode = resolveRootNode;
    return Promise.all([inChooser.initialize(inDiv), outChooser.initialize(outDiv)]);
  };

  /**
   * Loads the incoming and outgoing ORDs from the given Link or Component.
   *
   * @param {baja.Complex} value a parented `baja:Link` instance
   * to edit, or, if creating a brand new Link, a `Component` with `sourceOrd`
   * and `targetOrd` properties.
   * @returns {Promise} promise to be resolved with the requested ORDs
   * have been loaded
   */
  LinkEditor.prototype.doLoad = function (value) {
    var inChooser = this.$getIncomingOrdChooser(),
      outChooser = this.$getOutgoingOrdChooser();
    if (baja.hasType(value, 'baja:Component')) {
      if (!value.has('sourceOrd') || !value.has('targetOrd')) {
        throw new Error('sourceOrd and targetOrd properties required');
      }
      return Promise.all([inChooser.load(value.get('sourceOrd')), outChooser.load(value.get('targetOrd'))]);
    }
    if (!baja.hasType(value, 'baja:Link') || !value.getParent()) {
      throw new Error('parented baja:Link required');
    }
    var complex = value.getParent();
    return Promise.all([getSourceOrd(complex, value), getTargetOrd(complex, value)]).then(function (_ref) {
      var _ref2 = _slicedToArray(_ref, 2),
        sourceOrd = _ref2[0],
        targetOrd = _ref2[1];
      return Promise.all([inChooser.load(sourceOrd), outChooser.load(targetOrd)]);
    });
  };
  function resolveSourceOrd(sourceOrd, useHandles) {
    if (!useHandles) {
      return Promise.resolve(sourceOrd);
    } else {
      return sourceOrd.get({
        base: baja.station
      }).then(function (comp) {
        return baja.Ord.make('h:' + comp.getHandle());
      });
    }
  }

  /**
   * Reads the currently entered incoming and outgoing ORDs, and assembles them
   * into a `baja:Link` instance.
   *
   * @returns {Promise} promise to be resolved with the currently
   * configured `baja:Link`
   */
  LinkEditor.prototype.doRead = function () {
    var useHandles = this.properties().getValue('useHandles') !== false;
    return Promise.all([this.$getIncomingOrdChooser().read(), this.$getOutgoingOrdChooser().read()]).then(function (_ref3) {
      var _ref4 = _slicedToArray(_ref3, 2),
        inOrd = _ref4[0],
        outOrd = _ref4[1];
      //TODO: there has to be a nicer way to do this
      inOrd = inOrd.toString();
      outOrd = outOrd.toString();
      var ini = inOrd.lastIndexOf('/'),
        outi = outOrd.lastIndexOf('/'),
        sourceOrd = baja.Ord.make(inOrd.substr(0, ini)),
        sourceSlotName = inOrd.substr(ini + 1),
        targetSlotName = outOrd.substr(outi + 1);
      return resolveSourceOrd(sourceOrd, useHandles).then(function (sourceOrd) {
        return baja.$('baja:Link', {
          sourceOrd: sourceOrd,
          sourceSlotName: sourceSlotName,
          targetSlotName: targetSlotName
        });
      });
    });
  };
  return LinkEditor;
});
