/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

define(['underscore', 'Promise', 'nmodule/webEditors/rc/wb/mgr/MgrTypeInfo', 'nmodule/webEditors/rc/wb/mgr/commands/MgrCommand'], function (_, Promise, MgrTypeInfo, MgrCommand) {
  'use strict';

  var ACTION_BAR = MgrCommand.flags.ACTION_BAR;

  /**
   * Get the subjects from the selected rows in the given table.
   */
  function getSelectedRows(table) {
    return _.invoke(table.getSelectedRows(), 'getSubject');
  }

  /**
   * API Status: **Private**
   *
   * Utilities for working with bajaux Managers. The exported functions are currently intended
   * for internal manager subclasses only, for things that we don't necessarily want exposed on
   * the Manager API at the moment.
   *
   * @exports nmodule/webEditors/rc/wb/mgr/mgrUtils
   */
  var exports = {};
  var USER_DEFINED_NAME_KEY = 'mgrUtils.userDefinedName',
    HAS_DISCOVERY_NAME = 'mgrUtils.hasDiscoveryName',
    PROPOSALS_KEY = 'mgr:proposals';

  /**
   * Return the `ListSelection` instance for the main table of the manager.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} manager
   * @returns {module:nmodule/webEditors/rc/util/ListSelection}
   */
  exports.getMainTableSelection = function (manager) {
    return manager.getMainTable().$getSelection();
  };

  /**
   * Get an array of the currently selected subjects in the manager's main table.
   * For each of the selected rows, this will call the `Row`'s getSubject() function,
   * returning an array of the resulting subjects.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} manager
   * @Returns {Array.<*>}
   */
  exports.getMainTableSelectedSubjects = function (manager) {
    return getSelectedRows(manager.getMainTable());
  };

  /**
   * Return the `ListSelection` instance for the learn table of the manager.
   * The manager must have the `MgrLearn` mixin applied.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/mgr/MgrLearn} manager
   * @returns {module:nmodule/webEditors/rc/util/ListSelection}
   */
  exports.getLearnTableSelection = function (manager) {
    return manager.getLearnTable().$getSelection();
  };

  /**
   * Get an array of the currently selected subjects in the manager's discovery table.
   * For each of the selected rows, this will call the `Row`'s getSubject() function,
   * returning an array of the resulting subjects.The manager must have the `MgrLearn`
   * mixin applied.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/mgr/MgrLearn} manager
   * @Returns {Array.<*>}
   */
  exports.getLearnTableSelectedSubjects = function (manager) {
    return getSelectedRows(manager.getLearnTable());
  };

  /**
   * Find the first Command in the manager's command group by its constructor.
   *
   * @private
   * @since Niagara 4.14
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} manager
   * @param {Function} ctor the constructor of the required Command.
   * @returns {module:bajaux/commands/Command}
   */
  exports.findCommand = function (manager, ctor) {
    return _.find(manager.getCommandGroup().getChildren(), function (c) {
      return c instanceof ctor;
    });
  };

  /**
   * Find any Commands in the manager's command group by its constructor.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} manager
   * @param {Function} ctor the constructor of the required Command.
   * @returns {Array.<module:bajaux/commands/Command>}
   */
  exports.findCommands = function (manager, ctor) {
    return _.filter(manager.getCommandGroup().getChildren(), function (c) {
      return c instanceof ctor;
    });
  };

  /**
   * @private
   * @param {module:bajaux/commands/Command} cmd
   * @returns {boolean} true if this command should be shown in the action bar at the
   * botttom of the manager
   */
  exports.isShownInActionBar = function (cmd) {
    if (typeof cmd.$showInActionBar === 'boolean') {
      return cmd.$showInActionBar;
    }
    return !!(cmd.getFlags() & ACTION_BAR);
  };

  /**
   * Return an array of `MgrTypeInfo` instances for the given argument. If the
   * argument already contains one or more `MgrTypeInfo`s, these will be returned
   * unmodified. Any other type (e.g. string or baja.Type) will be passed to the
   * `MgrTypeInfo.make()` function. A non-array argument will resolve to an array
   * with a single `MgrTypeInfo` instance.
   *
   * @param {*} arg an array or single value to be converted to an array of `MgrTypeInfo`.
   *
   * @returns {Promise<Array.<module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo>>}
   */
  exports.toMgrTypeInfos = function (arg) {
    if (!_.isArray(arg)) {
      arg = [arg];
    }
    return Promise.all(arg.map(function (t) {
      return t instanceof MgrTypeInfo ? t : MgrTypeInfo.make(t);
    }));
  };

  /**
   * Called when adding a discovery item into the main table, this will take
   * the newly created component and `Row`, and set the proposed initial
   * values on it. This is done by passing the discovery node to the concrete
   * manager, and letting it return an object containing the proposed values,
   * based on whatever information it stored in the learn model.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/mgr/MgrLearn} mgr the manager
   * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} model the manager model
   * for the main table
   * @param {*} discovery the discovery object, obtained from the selected row
   * in the table of discovered objects.
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row the newly created Row instance
   * @param {boolean} [quick] if quick then commit without allowing proposed values to be cached for later updates
   *
   * @returns {Promise}
   */
  exports.setProposedDiscoveryValues = function (mgr, model, discovery, row, quick) {
    return Promise.resolve(mgr.getProposedValuesFromDiscovery(discovery, row.getSubject())).then(function (result) {
      var proposals = _.defaults(result, {
        values: {}
      });

      // keep the proposed name if it's already been changed in the editor
      if (exports.hasUserDefinedNameValue(row)) {
        proposals.name = exports.getProposedValue(row, '__name');
      }
      return Promise.all(Object.keys(proposals.values).map(function (prop) {
        var col = model.getColumn(prop);
        return Promise.resolve(col ? col.propose(proposals.values[prop], row) : Promise.reject(new Error('Unknown column ' + prop))).then(function () {
          return quick && col.commit(proposals.values[prop], row);
        });
      })).then(function () {
        var nameCol = model.getColumn('__name');
        if (nameCol) {
          var proposedName = proposals.hasOwnProperty('name') && proposals.name;
          exports.setHasDiscoveryName(row, !!proposedName);
          if (proposedName !== row.getSubject().getName()) {
            return nameCol.propose(proposals.name, row).then(function () {
              return quick && nameCol.commit(proposals.name, row);
            });
          }
        }
      });
    });
  };

  /**
   * Get the proposal keys for a specific row.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @returns {Array.<String>}
   * @since Niagara 4.14
   */
  exports.getProposalKeys = function (row) {
    var proposals = row.data(PROPOSALS_KEY);
    return Object.keys(proposals || []);
  };

  /**
   * Clear the existing proposals for a specific row.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @since Niagara 4.14
   */
  exports.clearProposals = function (row) {
    var proposals = row.data(PROPOSALS_KEY);
    if (proposals !== undefined) {
      row.deleteData(PROPOSALS_KEY);
    }
  };

  /**
   * Clear an existing proposal for a specific row and name.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {String} name
   * @since Niagara 4.14
   */
  exports.clearProposal = function (row, name) {
    var proposals = row.data(PROPOSALS_KEY);
    if (proposals) {
      delete proposals[name];
    }
  };

  /**
   * Get the existing proposal for a specific row and name.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {String} name
   * @returns {Object|undefined}
   * @since Niagara 4.14
   */
  exports.getProposedValue = function (row, name) {
    var proposals = row.data(PROPOSALS_KEY);
    return proposals && proposals[name];
  };

  /**
   * Set a proposal for a specific row and name.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {String} name
   * @param {*} [value] if no value is provided, the proposal will be set to undefined.
   * @since Niagara 4.14
   */
  exports.propose = function (row, name, value) {
    var proposals = row.data(PROPOSALS_KEY);
    if (proposals === undefined) {
      proposals = {};
      row.data(PROPOSALS_KEY, proposals);
    }
    proposals[name] = value;
  };

  /**
   * Mark the row as having a user defined name set. This is set
   * by the batch component editor. This is used to mark a row that
   * has had a proposed name explicitly typed by the user. It is
   * not meant to indicate a name proposed from a discovery item.
   * This is intended for internal manager framework use only.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {boolean} value
   * @since Niagara 4.14
   */
  exports.setHasUserDefinedNameValue = function (row, value) {
    row.data(USER_DEFINED_NAME_KEY, value);
  };

  /**
   * Test whether the given row has a user defined name proposed.
   * Note that this may return false for a row that has a name
   * proposed from a discovery item; this is only intended to
   * return true if a name has explicitly been typed by a user.
   * This is intended for internal manager framework use only.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @returns {boolean}
   * @since Niagara 4.14
   */
  exports.hasUserDefinedNameValue = function (row) {
    return !!row.data(USER_DEFINED_NAME_KEY);
  };

  /**
   * Mark the row as having a discovery name defined. This is set
   * by the `mgrUtils.setProposedDiscoveryValues` method. This is used to mark
   * a row that has had a proposed name explicitly returned by an override of the
   * `MgrLearn.getProposedValuesFromDiscovery` method.
   * This is intended for internal manager framework use only.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {boolean} value
   * @since Niagara 4.14
   */
  exports.setHasDiscoveryName = function (row, value) {
    row.data(HAS_DISCOVERY_NAME, value);
  };

  /**
   * Test whether the given row has a discovery name proposed.
   * This is intended for internal manager framework use only.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @returns {boolean}
   * @since Niagara 4.14
   */
  exports.hasDiscoveryName = function (row) {
    return !!row.data(HAS_DISCOVERY_NAME);
  };
  return exports;
});
