/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/options/MruOptions
 */
define(['underscore', 'nmodule/js/rc/switchboard/switchboard', 'nmodule/webEditors/rc/servlets/userData'], function (_, switchboard, userData) {
  'use strict';

  /**
   * We must switchboard this because multiple concurrent saves will overwrite
   * each other.
   *
   * @function
   * @param {module:nmodule/webEditors/rc/wb/options/MruOptions} mru
   * @returns {Promise}
   */
  var saveMru = switchboard(function (mru) {
    return getMru().then(function (obj) {
      obj = obj || {};
      obj[mru.getName()] = mru.getHistory();
      return putMru(obj);
    });
  }, {
    allow: 'oneAtATime'
  });

  /**
   * Container for most-recently-used (MRU) options to be used in various
   * different field editors. Local instances can be constructed, but typically
   * this data will be saved and retrieved from the server.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/options/MruOptions
   * @param {String} name unique name under which to store MRU history
   * @param {Array.<String>} [history] the MRU history
   * @param {Object} [params]
   * @param {number} [params.maxLength=15] maximum number of entries to store
   */
  var MruOptions = function MruOptions(name, history, params) {
    if (!name) {
      throw new Error('name required');
    }
    this.$name = name;
    this.$history = history || [];
    this.$maxLength = params && params.maxLength || 15;
  };

  /**
   * Retrieve MRU data from the server and construct a new `MruOptions`
   * instance. If no data exists on the server, the MRU history will simply be
   * empty.
   * @param {String} name
   * @param {Object} [params]
   * @param {baja.comm.Batch} [params.batch]
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/options/MruOptions>}
   */
  MruOptions.make = function (name, params) {
    return getMru(params).then(function (obj) {
      return new MruOptions(name, obj && obj[name] || [], params);
    });
  };

  /**
   * Adds a new value to the top of the history. Any entries past `maxLength`
   * will be evicted.
   * @param {String} str
   */
  MruOptions.prototype.add = function (str) {
    var history = this.$history,
      i;
    while ((i = _.indexOf(history, str)) >= 0) {
      history.splice(i, 1);
    }
    history.unshift(str);
    this.$history = history.slice(0, this.$maxLength);
  };

  /**
   * @returns {Array.<String>} the MRU history
   */
  MruOptions.prototype.getHistory = function () {
    return this.$history.slice();
  };

  /**
   * @returns {String} the MRU name
   */
  MruOptions.prototype.getName = function () {
    return this.$name;
  };

  /**
   * Save history data up to the server.
   *
   * @returns {Promise}
   */
  MruOptions.prototype.save = function () {
    return saveMru(this);
  };
  function getMru() {
    return userData.getJson('mru');
  }
  function putMru(obj) {
    return userData.putJson('mru', obj);
  }
  return MruOptions;
});
