/*
 * Copyright 2023 Tridium, Inc. All Rights Reserved.
 */

/* global niagara */

define(['baja!', 'log!nmodule.batchJob.rc.util.batchJobUtils', 'Promise', 'nmodule/webEditors/rc/fe/baja/util/spaceUtils', 'nmodule/webEditors/rc/wb/profile/profileUtils'], function (baja, log, Promise, spaceUtils, profileUtils) {
  'use strict';

  var logWarning = log.warning.bind(log);
  var resolveService = spaceUtils.resolveService;

  /**
   * API Status: **Private**
   * @exports nmodule/batchJob/rc/util/batchJobUtils
   * @since Niagara 4.14
   */
  var exports = {};

  /**
   * @returns {Promise.<baja.TimeZone>}
   */
  exports.getDisplayTz = function () {
    return baja.TimeZoneDatabase.get().then(function (db) {
      var displayTzId = profileUtils.getViewParameter('displayTzId');
      return displayTzId && db.getTimeZone(displayTzId) || db.getLocalTimeZone();
    });
  };

  /**
   * 
   * @param {String|baja.Ord} jobOrd 
   * @param {baja.TimeZone} [timezone] Timezone is only required for File Ords.
   * @returns {Promise}
   */
  exports.hyperlinkToJob = function (jobOrd, timezone) {
    var isFileOrd = String(jobOrd).indexOf('file:') > 0;
    if (isFileOrd) {
      // Workbench hyperlinking to Ux views does not like the local scheme for a file Ord.
      // Replacing that with station as the base seems like the right thing to do.
      // However, if we encounter more scenarios like this, Ord.normalize could be improved to do
      // this maybe?
      var fileOrd = String(jobOrd).split('local:|').pop();
      return niagara.env.hyperlink(baja.Ord.make({
        base: baja.Ord.make({
          base: 'station:',
          child: fileOrd
        }),
        child: new baja.ViewQuery('batchJob:BatchJobLogFileUxView?displayTzId=' + timezone.getId())
      }).normalize());
    }
    return niagara.env.hyperlink(baja.Ord.make(jobOrd));
  };

  /**
   * @returns {Promise.<baja.Value>}
   */
  exports.getBatchJobService = function () {
    return resolveService('batchJob:BatchJobService');
  };

  /**
   * Given a job start time and a base ord, get the correct job Ord
   * by making an RPC call to the server.
   * 
   * @param {baja.AbsTime|String} jobStartTime 
   * @param {baja.Ord|String} baseOrd 
   * @returns {Promise.<String>} the job Ord string which could reference a baja:Job component
   * under the JobService or the Job Log file.
   */
  exports.getJobOrd = function (jobStartTime, baseOrd) {
    return baja.rpc({
      typeSpec: "batchJob:BatchJobRpcUtil",
      method: "getJobOrd",
      args: [baja.hasType(jobStartTime, 'baja:AbsTime') ? jobStartTime.encodeToString() : String(jobStartTime), String(baseOrd)]
    });
  };

  /**
   * Gets the job details from a log file (.bjl)
   * @param {baja.file.File|baja.Ord|String} jobLogFile 
   * @returns {Promise.<baja.Component>} a baja:Component with 
   * "job" and "logContents" slots. The "job" slot is of type baja:Job and 
   * the "logContents" slot is the string encoded log contents 
   */
  exports.getJobDetailsFromLog = function (jobLogFile) {
    return baja.rpc({
      typeSpec: "batchJob:BatchJobRpcUtil",
      method: "getJobLogDetails",
      args: [baja.hasType(jobLogFile, 'baja:IFile') ? String(jobLogFile.getNavOrd()) : String(jobLogFile)]
    }).then(function (result) {
      return baja.bson.decodeAsync(JSON.parse(result));
    });
  };

  /**
   * Cancel one of more jobs. Maybe comparable (in pattern) to BatchJobService.dispose()
   * @param {baja.Value|Array.<baja.Value>} job
   * @return {Promise} 
   */
  exports.cancel = function (job) {
    if (!job) {
      return Promise.resolve();
    }
    var jobs = Array.isArray(job) ? job : [job];
    var mountedJobs = [],
      unmountedJobValues = [];
    jobs.forEach(function (jobValue) {
      if (baja.hasType(jobValue, 'baja:Job') && jobValue.isMounted()) {
        mountedJobs.push(jobValue);
      } else {
        unmountedJobValues.push(jobValue);
      }
    });
    return Promise["try"](function () {
      var proms = [];
      if (mountedJobs.length > 0) {
        proms.push(mountedJobs.map(function (mJob) {
          return mJob.cancel();
        }));
      }
      if (unmountedJobValues.length > 0) {
        // For looking up unmounted job we may base it off of the ProvisioningNwExt 
        var baseOrd = 'station:|slot:/Drivers/NiagaraNetwork/ProvisioningNwExt';
        proms.push(baja.rpc({
          typeSpec: "batchJob:BatchJobRpcUtil",
          method: "cancelJobs",
          args: [unmountedJobValues.map(function (job) {
            return baja.hasType(job, 'batchJob:IBatchJobSummary') ? job.getTimestamp().encodeToString() : job.getStartTime().encodeToString();
          }), baseOrd]
        }));
      }
      return Promise.all(proms);
    })["catch"](logWarning);
  };
  return exports;
});
