/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/videoDriver/rc/fe/playback/PlaybackMixin
 */
define(['Promise', 'underscore', 'nmodule/js/rc/tinyevents/tinyevents', 'nmodule/webEditors/rc/wb/mixin/mixinUtils'], function (Promise, _, tinyevents, mixinUtils) {
  'use strict';

  var applyMixin = mixinUtils.applyMixin;
  var isFunction = _.isFunction;
  var MIXIN_NAME = 'videoDriver:PlaybackMixin';

  /**
   * Interface for an object capable of performing playback controls on a camera
   * such as play/pause/rewind.
   * @interface module:nmodule/videoDriver/rc/fe/playback/PlaybackController
   */

  /**
   * Changes the current playback mode. If any argument is `undefined` it
   * indicates no change; it is up to the implementor to maintain its current
   * playback state and only change one aspect at a time.
   *
   * @function playback
   * @memberOf module:nmodule/videoDriver/rc/fe/playback/PlaybackController
   * @param {object} [params] (see BPlaybackControlInfo)
   * @param {string} [params.type] a `BPlaybackTypeEnum` tag
   * @param {number} [params.speed] playback speed, from 1 to 10. For normal
   * rewind and fast-forward, 1 is "just a bit faster" and 10 is "max speed".
   * For slow-rewind and slow-forward, 1 is "just a bit slower" and 10 is
   * "as slow as we can reasonably go." It's up to the implementer to decide the
   * actual playback speeds. For play, pause, and live, the speed should have
   * no effect.
   * @param {number} [params.timestamp] instant to start playback
   */

  /**
   * @function getCamera
   * @memberOf module:nmodule/videoDriver/rc/fe/playback/PlaybackController
   * @returns {baja.Component}
   */

  /**
   * If this function is available, it can provide guidance on the last known timestamp that was played.
   * @function getCurrentTimestamp
   * @memberOf module:nmodule/videoDriver/rc/fe/playback/PlaybackController
   * @returns {Date}
   */

  /**
   * This mixin should be applied to a `VideoStream` to mark it as supporting
   * playback controls such as play/pause/rewind.
   *
   * In order for it to be successfully applied, the `VideoStream` must
   * implement `doPlayback` and `getCamera` functions. The signature for
   * `doPlayback` is the same as for `playback` - implementing it as a separate
   * function allows for playback events to be fired. When calling playback
   * functions programmatically, call `playback`.
   *
   * Implement `doPlayback`, call `playback`: just like a bajaux Widget.
   *
   * @mixin
   * @mixes tinyevents
   * @alias module:nmodule/videoDriver/rc/fe/playback/PlaybackMixin
   * @implements module:nmodule/videoDriver/rc/fe/playback/PlaybackController
   */
  var PlaybackMixin = function PlaybackMixin(controller) {
    if (!controller) {
      throw new Error('controller required');
    }
    if (!isFunction(controller.doPlayback)) {
      throw new Error('doPlayback function required');
    }
    if (!isFunction(controller.getCamera)) {
      throw new Error('getCamera function required');
    }
    if (!applyMixin(controller, MIXIN_NAME, PlaybackMixin.prototype)) {
      return;
    }
    tinyevents(controller);
  };
  PlaybackMixin.prototype.playback = function (params) {
    this.emit(PlaybackMixin.PLAYBACK_CHANGED_EVENT, params);
    return Promise.resolve(this.doPlayback(params));
  };
  PlaybackMixin.PLAYBACK_CHANGED_EVENT = 'videoDriver:playback';
  return PlaybackMixin;
});
