/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/* global niagara */

/**
 * API Status: **Private**
 * @module nmodule/driver/rc/wb/mgr/columns/DeviceExtsMgrColumn
 */
define(['baja!', 'lex!driver', 'underscore', 'nmodule/webEditors/rc/wb/mgr/model/columns/IconMgrColumn', 'baja!driver:DeviceExt'], function (baja, lexs, _, IconMgrColumn) {
  'use strict';

  var driverLex = lexs[0];
  function getDeviceExts(comp) {
    return comp.getSlots(function (slot) {
      return (comp.getFlags(slot) & baja.Flags.HIDDEN) === 0;
    }).properties().is('driver:DeviceExt').toValueArray();
  }

  /**
   * Column type for the BDeviceExts on a BDevice. Each extension is represented
   * by an icon that, when clicked, will hyperlink to the default view for the
   * extension. Extensions can be pre-determined at the column construction
   * time, using a prototype component instance, or can be found dynamically for
   * each row at the time the cell is constructed.
   *
   * @class
   * @alias module:nmodule/driver/rc/wb/mgr/columns/DeviceExtsMgrColumn
   * @extends module:nmodule/webEditors/rc/wb/mgr/model/columns/IconMgrColumn
   * @param {String} name - the column name
   * @param {Object} [params]
   * @param {baja.Component} [params.component] - a prototype component, used
   * when not looking for exts dynamically.
   * @param {Boolean} [params.dynamic] - a boolean indicating that exts should
   * be looked for on every individual component in the row, not the prototype
   * component.
   * @throws {Error} if dynamic = false and prototype component is not given.
   */
  var DeviceExtsMgrColumn = function DeviceExtsMgrColumn(name, params) {
    params = params || {};
    IconMgrColumn.call(this, name, _.defaults(params, {
      displayName: driverLex.get('deviceExtsColumn')
    }));
    this.$dynamic = !!params.dynamic;
    if (!this.$dynamic) {
      if (params.component) {
        this.$exts = getDeviceExts(params.component);
      } else {
        throw new Error('must provide a prototype if not using dynamic extension lookup');
      }
    }
    this.setHidable(true);
  };
  DeviceExtsMgrColumn.prototype = Object.create(IconMgrColumn.prototype);
  DeviceExtsMgrColumn.prototype.constructor = DeviceExtsMgrColumn;

  /**
   * Return the value for the row. If the column is configured to look for
   * dynamic device extensions, query for those now.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @returns {Array.<baja.Component|*>}
   */
  DeviceExtsMgrColumn.prototype.getValueFor = function (row) {
    var subject = row.getSubject();
    if (baja.hasType(subject, 'driver:DeviceFolder')) {
      return [];
    }
    return this.$dynamic ? getDeviceExts(subject) : this.$exts;
  };

  /**
   * Hyperlink to the ext's view when the icon is clicked.
   *
   * @param {baja.Ord} ord - the ord to hyperlink to.
   */
  DeviceExtsMgrColumn.prototype.doHyperlink = function (ord) {
    if (ord && typeof niagara !== 'undefined') {
      niagara.env.hyperlink(ord);
    }
  };

  /**
   * Build the cell's content. For each icon editor representing a device
   * ext, create a click handler to hyperlink to the extension's ord.
   *
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {jQuery} dom
   * @returns {*}
   */
  DeviceExtsMgrColumn.prototype.buildCell = function (row, dom) {
    var that = this,
      subject = row.getSubject();
    return IconMgrColumn.prototype.buildCell.apply(that, arguments).then(function (eds) {
      _.each(eds, function (ed) {
        var jq = ed.jq();
        jq.on('click', function (e) {
          var ext, ord;

          // We need to test the class of the row the cell has been added to. If
          // it is in the main manager table, we should perform the hyperlink. If
          // the cell is in the table for the batch component editor, ignore the
          // click.

          if (jq.closest('tr').hasClass('driver-mgr-row') && e.button !== 3) {
            ext = ed.value();
            ord = that.$dynamic ? ext.getNavOrd() : baja.Ord.make(subject.getNavOrd() + '/' + ext.getName());
            that.doHyperlink(ord.relativizeToSession());
            return false;
          }
          return true;
        });
      });
    });
  };
  return DeviceExtsMgrColumn;
});
//# sourceMappingURL=DeviceExtsMgrColumn.js.map
