/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*global setTimeout: false, clearTimeout: false */

/**
 * API Status: **Private**
 * @module nmodule/js/rc/asyncUtils/promiseMux
 */
define([], function () {
  'use strict';

  function defaultDeferred() {
    var obj = {},
      Promise = require('Promise'),
      // eslint-disable-next-line promise/avoid-new
      promise = new Promise(function (resolve, reject) {
        obj.resolve = resolve;
        obj.reject = reject;
      });
    obj.promise = function () {
      return promise;
    };
    return obj;
  }

  /**
   * Create a function that "multiplexes" Promises together.
   *
   * Quite often there will be a case where async operations can be batched
   * together on the backend, but the caller of your function doesn't want to
   * have to worry about the batching himself/herself. This allows you to define
   * the batching behavior, but have the caller continue to make individual
   * requests, ignorant of the backend batching. Requests to this function will
   * introduce a small delay, allowing multiple requests to "pile up" before
   * being sent to the async service in one go.
   *
   * @alias module:nmodule/js/rc/asyncUtils/promiseMux
   * @param {Object} params
   * @param {Function} params.exec a function that will perform an async
   * operation on an array of arguments (each argument is what was passed to
   * one call of the mux'ed function). This function may return a promise, and
   * must resolve to an array of results of the same length as the array of
   * arguments.
   * @param {Number} [params.delay=0] introduce a delay of this many ms. The
   * resulting function will be "debounced" so that `exec` will only be invoked
   * after that many ms after the last time the function was called.
   * @param {Boolean|String} [params.cache=false] set this to true to cause
   * the results of promises to be held onto in memory, avoiding all further
   * network calls for that argument. Use this when the result of your async
   * operation will never change at runtime for a given input. Note that
   * rejected promises will always be retried regardless of the cache setting.
   * @param {Boolean} [params.coalesce=true] by default, arguments are treated
   * as "unique" - identical arguments will just be coalesced together into the
   * same backend call. Set this to `false` to cause every call to the function
   * result in another argument passed to `exec`. (In order to be considered
   * identical, two arguments should `toString` to the same value.)
   * @param {Function} [params.onQueue] a function to be called when the queue
   * is modified, potentially forcing the mux'ed function to execute if the
   * queue has grown too large. See example.
   * @param {Function} [params.deferred] if you're using your own promise
   * library, you can pass a function to return a new deferred object:
   * `{ resolve: Function, reject: Function, promise: Function }`.
   * @returns {Function} a mux'ed function.
   *
   * @example
   *   <caption>Use `coalesce=true` when identical arguments shouldn't result in
   *   any extra network overhead.</caption>
   *
   *   // importing baja:String twice isn't going to provide me any extra
   *   // useful information.
   *
   *   var importOneType = promiseMux({
   *     exec: function (typeSpecs) {
   *       return baja.importTypes(typeSpecs);
   *     },
   *     delay: 20,
   *     coalesce: true //or undefined
   *   });
   *
   *   // same results as baja.importTypes([ 'baja:String', 'baja:Boolean' ])
   *   // since identical arguments get coalesced - but all calls will be still
   *   // be resolved with the correct value.
   *
   *   Promise.all([
   *     importOneType('baja:String'),
   *     importOneType('baja:Boolean'),
   *     importOneType('baja:String')
   *   ]).then(function (types) {
   *     types.join() === 'baja:String,baja:Boolean,baja:String';
   *   });
   *
   * @example
   *   <caption>Use `coalesce=false` when you still want the network batching,
   *   but identical calls are still useful information.</caption>
   *
   *   var logOneMessage = promiseMux({
   *     exec: function (messages) {
   *       return sendMessagesToServer(messages);
   *     },
   *     delay: 20,
   *     coalesce: false
   *   });
   *
   *   // each individual message will make it to the server, but all get
   *   // batched into one network call. same as:
   *   // sendMessagesToServer([ 'hello', 'world', 'hello', 'world' ]);
   *
   *   Promise.all([
   *     logOneMessage('hello'),
   *     logOneMessage('world'),
   *     logOneMessage('hello'),
   *     logOneMessage('world')
   *   ]);
   *   
   * @example
   * <caption>Use onQueue to force the mux'ed function to execute if the queue
   * has gotten too large.</caption>
   *   var MAX_TYPES_TO_IMPORT_AT_ONCE = 10;
   *   var importOneType = promiseMux({
   *     exec: function (typeSpecs) {
   *       return baja.importTypes(typeSpecs);
   *     },
   *     onQueue: function (typeSpec, queue, exec) {
   *       //queue will be an object if coalesce = true, an array if coalesce = false.
   *       //force the mux'ed function to execute and clear out the queue before
   *       //continuing.
   *       if (Object.keys(queue).length >= MAX_TYPES_TO_IMPORT_AT_ONCE) {
   *         return exec();
   *       }
   *     },
   *     coalesce: true
   *   });
   */
  var promiseMux = function promiseMux(params) {
    var exec = params && params.exec;
    if (typeof exec !== 'function') {
      promiseMux.$throwError('exec function required');
    }
    var delay = params.delay;
    if (delay && delay < 0) {
      promiseMux.$throwError('delay cannot be less than 0');
    }
    var deferred = params.deferred || defaultDeferred,
      onQueue = params.onQueue || noop;
    var keyMap = {},
      deferredMap = {},
      coalesce = params.coalesce !== false,
      cache = coalesce && params.cache,
      argsArray = [],
      dfArray = [],
      performTicket = null;
    var perform = function perform() {
      var deferredKeys = [],
        dfs = [],
        args = [];
      if (coalesce) {
        Object.keys(deferredMap).forEach(function (key) {
          var df = deferredMap[key];
          if (!df.started) {
            df.started = true;
            deferredKeys.push(key);
            dfs.push(df);
            args.push(keyMap[key]);
          }
        });
      } else {
        args = argsArray;
        dfs = dfArray;
        argsArray = [];
        dfArray = [];
      }
      return resolve(exec(args), deferred).then(function (results) {
        results.forEach(function (result, i) {
          if (!dfs[i]) {
            promiseMux.$throwError("exec callback resolved to the wrong number of Promises: " + results.length + " != " + dfs.length);
          }
          dfs[i].resolve(result);
          if (!cache) {
            delete deferredMap[deferredKeys[i]]; //can call for this arg again
            delete keyMap[deferredKeys[i]];
          }
        });
        if (results.length < dfs.length) {
          promiseMux.$throwError("exec callback resolved to the wrong number of Promises: " + results.length + " != " + dfs.length);
        }
      }, function (err) {
        dfs.forEach(function (df, i) {
          df.reject(err);
          delete deferredMap[deferredKeys[i]]; //can call for this arg again
          delete keyMap[deferredKeys[i]];
        });
      });
    };
    var mux = function mux(arg) {
      var df, onQueuePromise;
      if (coalesce) {
        df = deferredMap[arg];
        if (df) {
          return df.promise();
        }
      }
      clearTimeout(performTicket);
      performTicket = setTimeout(perform, delay);
      df = deferred();
      if (coalesce) {
        onQueuePromise = resolve(onQueue(arg, clone(keyMap), perform), deferred);
        keyMap[arg] = arg;
        deferredMap[arg] = df;
      } else {
        onQueuePromise = resolve(onQueue(arg, argsArray.slice(), perform), deferred);
        argsArray.push(arg);
        dfArray.push(df);
      }
      return onQueuePromise.then(function () {
        return df.promise();
      });
    };
    return mux;
  };
  function resolve(obj, deferred) {
    var df = deferred();
    df.resolve(obj);
    return df.promise();
  }
  function clone(obj) {
    return Object.keys(obj).reduce(function (o, key) {
      o[key] = obj[key];
      return o;
    }, {});
  }
  function noop() {}

  /**
   * @private
   * @param {String} message
   * @throws {Error}
   */
  promiseMux.$throwError = function (message) {
    throw new Error(message);
  };
  return promiseMux;
});
