function _slicedToArray(r, e) { return _arrayWithHoles(r) || _iterableToArrayLimit(r, e) || _unsupportedIterableToArray(r, e) || _nonIterableRest(); }
function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
function _iterableToArrayLimit(r, l) { var t = null == r ? null : "undefined" != typeof Symbol && r[Symbol.iterator] || r["@@iterator"]; if (null != t) { var e, n, i, u, a = [], f = !0, o = !1; try { if (i = (t = t.call(r)).next, 0 === l) { if (Object(t) !== t) return; f = !1; } else for (; !(f = (e = i.call(t)).done) && (a.push(e.value), a.length !== l); f = !0); } catch (r) { o = !0, n = r; } finally { try { if (!f && null != t["return"] && (u = t["return"](), Object(u) !== u)) return; } finally { if (o) throw n; } } return a; } }
function _arrayWithHoles(r) { if (Array.isArray(r)) return r; }
/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/wiresheet/rc/wb/layout/Mask
 */
define(['underscore', 'nmodule/wiresheet/rc/wb/WbConstants', 'nmodule/wiresheet/rc/wb/util/wsUtils'], function (_, WbConstants, wsUtils) {
  'use strict';

  var allPointsInBox = wsUtils.allPointsInBox,
    segmentToBox = wsUtils.segmentToBox,
    toDirection = wsUtils.toDirection,
    translate = wsUtils.translate;
  var LINK_H = WbConstants.LINK_H,
    LINK_V = WbConstants.LINK_V,
    LINK_L2U = WbConstants.LINK_L2U,
    LINK_L2D = WbConstants.LINK_L2D,
    LINK_R2U = WbConstants.LINK_R2U,
    LINK_R2D = WbConstants.LINK_R2D,
    LINK_STUB_T = WbConstants.LINK_STUB_T,
    LINK_STUB_S = WbConstants.LINK_STUB_S,
    COMPONENT_OUT = WbConstants.COMPONENT_OUT,
    COMPONENT_IN = WbConstants.COMPONENT_IN,
    WIXEL_PADDING = WbConstants.WIXEL_PADDING,
    USABLE_MIN_DIMENSION = WbConstants.USABLE_MIN_DIMENSION;
  var joints = {
    'RU': LINK_L2U,
    'RD': LINK_L2D,
    'UL': LINK_L2D,
    'UR': LINK_R2D,
    'LU': LINK_R2U,
    'LD': LINK_R2D,
    'DL': LINK_L2U,
    'DR': LINK_R2U
  };

  /**
   * A `Mask` is a representation of the contents of a wiresheet at each wixel.
   *
   * Each wixel can hold a collection of tiles, one per glyph that occupies that
   * wixel. An empty wixel holds no tiles.
   *
   * Each tile holds a reference to the entity occupying that wixel (a
   * `ComponentGlyph`, `SnakeGlyph`, etc.) and a numeric mask value
   * indicating the visual contents of that wixel (a horizontal segment,
   * vertical segment, a wixel in a component, etc. - these values reside in
   * `WbConstants`). By bitwise-ORing these mask values together you can get an
   * idea of the overall layered visual contents of the wiresheet at that wixel.
   *
   * @class
   * @alias module:nmodule/wiresheet/rc/wb/layout/Mask
   * @param {Object} params
   * @param {number} params.maxWidth the max width in wixels
   * @param {number} params.maxHeight the max height in wixels
   * @param {number} [params.minWidth=0] the initial minimum width in wixels.
   * This will used as the width when the mask is not occupied by any entities.
   * @param {number} [params.minHeight=0] the initial minimum height in wixels.
   * This will used as the height when the mask is not occupied by any entities.
   * @param {Array.<Object>} [params.entities] an array of glyph entities with
   * which to initially populate the mask
   */
  var Mask = function Mask(_ref) {
    var maxWidth = _ref.maxWidth,
      maxHeight = _ref.maxHeight,
      _ref$minHeight = _ref.minHeight,
      minHeight = _ref$minHeight === void 0 ? 0 : _ref$minHeight,
      _ref$minWidth = _ref.minWidth,
      minWidth = _ref$minWidth === void 0 ? 0 : _ref$minWidth,
      _ref$entities = _ref.entities,
      entities = _ref$entities === void 0 ? [] : _ref$entities;
    if (maxWidth === undefined) {
      throw new Error('maxWidth required');
    }
    if (maxHeight === undefined) {
      throw new Error('maxHeight required');
    }
    this.$minHeight = minHeight;
    this.$minWidth = minWidth;
    this.setMaxWidth(maxWidth);
    this.setMaxHeight(maxHeight);
    this.$topX = 0;
    this.$topY = 0;
    this.$generateBoard(entities);
  };

  /**
   * Erases all entities from the mask.
   */
  Mask.prototype.clear = function () {
    var $maxWidth = this.$maxWidth,
      $maxHeight = this.$maxHeight;
    this.$topX = 0;
    this.$topY = 0;
    this.$board = _.map(_.range(0, $maxHeight), function () {
      var arr = [];
      arr.length = $maxWidth;
      return arr;
    });
  };
  Mask.prototype.$generateBoard = function (entities) {
    this.clear();
    _.each(entities, this.addEntity.bind(this));
  };

  /**
   * Add an entity reference to this mask.
   * @param {{ glyph: Object, layout: Object}} entity
   */
  Mask.prototype.addEntity = function (entity) {
    if (!entity.layout) {
      return;
    }
    switch (entity.glyph.type) {
      case 'ComponentGlyph':
        return componentGlyph(this, entity);
      case 'StubGlyph':
        return stubGlyph(this, entity);
      case 'SnakeGlyph':
        return snakeGlyph(this, entity);
      case 'TextBlockGlyph':
        return componentGlyph(this, entity);
    }
  };

  /**
   * @returns {number} the height of the mask in wixels
   */
  Mask.prototype.getHeight = function () {
    var topY = this.$topY,
      height = topY + WIXEL_PADDING,
      maxHeight = this.$maxHeight,
      minHeight = this.$minHeight;
    if (topY === 0) {
      return minHeight;
    }
    if (height > maxHeight) {
      return maxHeight;
    }
    if (height < minHeight) {
      return minHeight;
    }
    return height;
  };

  /**
   * @returns {number} the width of the mask in wixels
   */
  Mask.prototype.getWidth = function () {
    var topX = this.$topX,
      width = this.$topX + WIXEL_PADDING,
      maxWidth = this.$maxWidth,
      minWidth = this.$minWidth;
    if (topX === 0) {
      return minWidth;
    }
    if (width > maxWidth) {
      return maxWidth;
    }
    if (width < minWidth) {
      return minWidth;
    }
    return width;
  };

  /**
   * Sets the minimum width
   *
   * @param {number} minWidth the wiresheets minimum width in wixels.
   */
  Mask.prototype.setMinWidth = function (minWidth) {
    this.$minWidth = minWidth;
  };

  /**
   * Sets the minimum height
   *
   * @param {number} minHeight the wiresheets minimum height in wixels.
   */
  Mask.prototype.setMinHeight = function (minHeight) {
    this.$minHeight = minHeight;
  };

  /**
   * Sets the maximum width
   *
   * @param {Number} maxWidth the wiresheets maximum width in wixels.
   */
  Mask.prototype.setMaxWidth = function (maxWidth) {
    this.$maxWidth = Math.max(maxWidth, this.$getUsableMinDimension());
  };

  /**
   * Sets the maximum height
   *
   * @param {Number} maxHeight the wiresheets maximum height in wixels.
   */
  Mask.prototype.setMaxHeight = function (maxHeight) {
    this.$maxHeight = Math.max(maxHeight, this.$getUsableMinDimension());
  };

  /**
   * Provides distance in wixels from the top left corner to the last wixel that
   * has an entity to display in both the x and y directions which is returned
   * as the width and height respectively.
   *
   * @returns {{ width: number, height: number }}
   */
  Mask.prototype.getOccupiedSize = function () {
    return {
      width: this.$topX,
      height: this.$topY
    };
  };

  /**
   * Get a collection of all tiles occupying this wixel.
   * @param {number} x
   * @param {number} y
   * @returns {Array.<{tile: number, entity: Object}>}
   */
  Mask.prototype.getTiles = function (x, y) {
    var obj = getObj(this, x, y);
    return obj ? obj.tiles.slice() : [];
  };

  /**
   * Get a mask value representing the layered visual contents of the wiresheet
   * at this wixel, derived by bitwise-ORing all the individual tile values
   * together.
   * @param {number} x
   * @param {number} y
   * @returns {number}
   */
  Mask.prototype.getTileMask = function (x, y) {
    var tiles = _.pluck(this.getTiles(x, y), 'tile');
    return _.reduce(tiles, function (memo, tile) {
      return memo | tile;
    }, 0);
  };
  Mask.prototype.$getUsableMinDimension = function () {
    return USABLE_MIN_DIMENSION;
  };
  function componentGlyph(mask, entity) {
    var _entity$layout = entity.layout,
      x = _entity$layout.x,
      y = _entity$layout.y,
      width = _entity$layout.width,
      height = _entity$layout.height;
    for (var ry = 0; ry < height; ++ry) {
      for (var rx = 0; rx < width; ++rx) {
        var ax = rx + x,
          ay = ry + y,
          tile = !rx || !ry || rx === width - 1 || ry === height - 1 ? COMPONENT_OUT : COMPONENT_IN;
        addTile(mask, ax, ay, tile, entity);
      }
    }
  }
  function stubGlyph(mask, entity) {
    var _entity$layout2 = entity.layout,
      x = _entity$layout2.x,
      y = _entity$layout2.y,
      width = _entity$layout2.width,
      height = _entity$layout2.height,
      tile = entity.glyph.direction === 'in' ? LINK_STUB_T : LINK_STUB_S;
    for (var ry = 0; ry < height; ++ry) {
      for (var rx = 0; rx < width; ++rx) {
        var ax = rx + x,
          ay = ry + y;
        addTile(mask, ax, ay, tile, entity);
      }
    }
  }
  function snakeGlyph(mask, entity) {
    var glyph = entity.glyph;
    if (glyph.segments) {
      applySnakeSegments(mask, entity);
    } else if (glyph.stubs) {
      applySnakeStubs(mask, entity);
    }
  }
  function applySnakeSegments(mask, entity) {
    var segments = entity.glyph.segments,
      layout = entity.layout;
    segments = segments.map(translate.bind(null, layout));
    var pairs = _.reduce(segments, function (memo, cur, i) {
      return i ? memo.concat([{
        p1: segments[i - 1],
        p2: cur
      }]) : [];
    }, []);

    // special case: when the first or last segment is 0 length, consider first
    // point to be 1 wixel left or second point 1 wixel right for direction
    // calculation. see "moves breakpoint all the way" SimpleRoutingStrategy
    // specs for examples.
    var direction = function direction(_ref2, i) {
      var p1 = _ref2.p1,
        p2 = _ref2.p2;
      if (p1.x === p2.x && p1.y === p2.y) {
        if (i === 0) {
          p1 = {
            x: p1.x - 1,
            y: p1.y
          };
        } else if (i === pairs.length - 1) {
          p2 = {
            x: p2.x + 1,
            y: p2.y
          };
        }
      }
      return toDirection(p1, p2);
    };
    _.each(pairs, function (pair, i) {
      var prev = pairs[i - 1],
        dir = direction(pair, i),
        prevDir = prev && direction(prev, i - 1),
        points = allPointsInSegment(pair);

      // for all segments but the last, skip adding the final point - otherwise
      // we double-count at the joints.
      if (i < pairs.length - 1) {
        points = _.initial(points);
      }
      _.each(points, function (_ref3, i) {
        var x = _ref3.x,
          y = _ref3.y;
        var tile;
        if (prev && !i) {
          tile = joints[prevDir + dir];
        } else {
          tile = dir === 'U' || dir === 'D' ? LINK_V : LINK_H;
        }
        addTile(mask, x, y, tile, entity);
      });
    });
  }
  function applySnakeStubs(mask, entity) {
    var _entity$layout3 = entity.layout,
      x = _entity$layout3.x,
      y = _entity$layout3.y;
    var _entity$glyph$stubs = _slicedToArray(entity.glyph.stubs, 2),
      sourceStub = _entity$glyph$stubs[0],
      targetStub = _entity$glyph$stubs[1];
    addTile(mask, x + sourceStub.x, y + sourceStub.y, LINK_STUB_S, entity);
    addTile(mask, x + targetStub.x, y + targetStub.y, LINK_STUB_T, entity);
  }
  function addTile(mask, x, y, tile, entity) {
    var obj = getObj(mask, x, y);
    if (obj) {
      mask.$topX = Math.max(mask.$topX, x + 1);
      mask.$topY = Math.max(mask.$topY, y + 1);
      pushToFrontOfTiles(obj.tiles, tile, entity);
    }
  }
  function pushToFrontOfTiles(tiles, tile, entity) {
    var id = entity.__id;
    var index = _.findIndex(tiles, function (t) {
      return t.entity.__id === id;
    });
    if (index >= 0) {
      tiles.splice(index, 1);
    }
    tiles.unshift({
      tile: tile,
      entity: entity
    });
  }
  function getObj(_ref4, x, y) {
    var $maxWidth = _ref4.$maxWidth,
      $maxHeight = _ref4.$maxHeight,
      $board = _ref4.$board;
    if (x >= 0 && x < $maxWidth && y >= 0 && y < $maxHeight) {
      return $board[y][x] || ($board[y][x] = {
        tiles: []
      });
    }
  }
  function allPointsInSegment(_ref5) {
    var p1 = _ref5.p1,
      p2 = _ref5.p2;
    var dir = toDirection(p1, p2),
      points = allPointsInBox(segmentToBox(p1, p2));
    if (dir === 'L' || dir === 'U') {
      points.reverse();
    }
    return points;
  }
  return Mask;
});
