/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * RelativeContributionChart.js
 * This javascript creates a Relative Contribution Chart using d3 javascript.
 */
define(['baja!', 'bajaux/Widget', 'bajaux/mixin/subscriberMixIn', 'Promise', 'jquery', 'd3', 'underscore', 'nmodule/analytics/rc/chart/base/AnalyticC3BaseChart', 'nmodule/analytics/rc/util/analyticsUtil', 'nmodule/analytics/rc/chart/fe/AnalyticRankingChartUXSettingsFE', 'nmodule/webEditors/rc/wb/util/ICollectionSupport', 'lex!analytics', 'css!nmodule/analytics/rc/chart/relcontrib/relContribChart', "baja!analytics:C3UxNoAxisWebChartParams"], function (baja, Widget, subscriberMixIn, Promise, $, d3, _, AnalyticC3BaseChart, analyticsUtil, AnalyticRankingChartUXSettingsFE, ICollectionSupport, lexicon) {
  "use strict";

  var lex = lexicon[0];

  /**
   * RelativeContributionChart shows the contribution of each of the bindings to the overall sum total of
   * the selected binding list
   *
   * @class
   * @alias module:nmodule/analytics/rc/chart/relcontrib/RelativeContributionChart
   * @extends module:nmodule/analytics/rc/chart/base/AnalyticC3BaseChart
   * @mixes module:nmodule/webEditors/rc/wb/util/ICollectionSupport
   * @param {Object} [params]
   */
  var RelativeContributionChart = function RelativeContributionChart(params) {
    AnalyticC3BaseChart.call(this, $.extend({}, params));
    ICollectionSupport(this);
    this.columnNames = ['Series Name', 'Value', 'Percentage(%)', lex.get("chart.interpolationStatus.label")];
    this.columnKeys = ['x', 'val', 'y', "metaData.interpolationStatus"];
    this.textLengthLimitScale = d3.scale.threshold().domain(this.supportedResolutions).range([5, 7, 15, 17, 19, 25, 30]);
  };
  RelativeContributionChart.prototype = Object.create(AnalyticC3BaseChart.prototype);
  RelativeContributionChart.prototype.constructor = RelativeContributionChart;

  /**
   * Render the pie
   * @param chartSettingsCollection
   */
  RelativeContributionChart.prototype.processData = function (chartSettingsCollection, isRedrawRequest) {
    var that = this;
    return new Promise(function (resolve, reject) {
      var pieData = [];
      var total = 0;
      // Loop over the collection and set the contribution percentage.
      _.map(chartSettingsCollection, function (model, index) {
        var data = model.getAnalyticTrendArray();
        if (!isRedrawRequest) {
          data = analyticsUtil.normalize(model, data, true);
        }
        var contrib = d3.sum(data, function (e) {
          return e.value;
        });
        var finalTrendFlags = 0;
        data.map(function (d) {
          finalTrendFlags = finalTrendFlags | d.trendFlags;
        });
        total += contrib;
        pieData.push({
          baseOrd: analyticsUtil.getBaseOrd(model.getOrd()),
          seriesName: baja.SlotPath.unescape(model.getSeriesName()),
          value: contrib,
          brush: model.getBrush(),
          unit: model.getUnitDisplay(),
          ord: model.getOrd(),
          chartType: model.getChartType(),
          showOnY2: model.getShowOnY2(),
          metaData: {
            interpolationStatus: analyticsUtil.decodeTrendFlag(finalTrendFlags)
          }
        });
      });

      // Sort the pie data by value ascending.
      pieData = _.sortBy(pieData, function (e) {
        return e.value;
      }).reverse();
      that.series = $.map(pieData, function (rec) {
        var inOb = {};
        // name: rec.baseOrd,
        if (total > 0 && rec.value > 0) {
          inOb = {
            name: rec.seriesName,
            data: [{
              x: rec.seriesName,
              // yVal: rec.value,
              y: rec.value * 100 / total,
              val: rec.value,
              metaData: rec.metaData
            }],
            color: rec.brush,
            unitSymbol: rec.unit,
            unit: rec.unit,
            random: Math.random(),
            chartType: rec.chartType,
            showOnY2: rec.showOnY2
          };
        } else {
          inOb = {
            name: rec.seriesName,
            data: [{
              x: rec.seriesName,
              // yVal: rec.value,
              y: lex.get("time.operation.minus"),
              val: lex.get("time.operation.minus"),
              metaData: rec.metaData
            }],
            color: rec.brush,
            unitSymbol: rec.unit,
            unit: rec.unit,
            random: Math.random(),
            chartType: rec.chartType,
            showOnY2: rec.showOnY2
          };
        }
        return inOb;
      });
      resolve(analyticsUtil.getSimpleC3Data(that, that.series));
    });
  };

  /**
   * Get the chart margins
   * @type {{top: number, right: number, bottom: number, left: number}}
   */
  RelativeContributionChart.prototype.chartMargins = function () {
    return {
      top: 35,
      right: 70,
      bottom: 35,
      left: 70
    };
  };
  /**
   * This needs a clean slate
   * @returns {boolean}
   */
  RelativeContributionChart.prototype.needsCleanSlate = function () {
    return true;
  };

  /**
   * This method should be overridden by subclasses to return respective chart types.
   * The default implementation retruns "none".
   * @returns {string}
   */
  RelativeContributionChart.prototype.getSupportedExportTypes = function () {
    return ["rcchart", "csv"];
  };
  RelativeContributionChart.prototype.getXAxisLabel = function () {
    return lex.get("chart.aggregation.xaxis.label");
  };
  RelativeContributionChart.prototype.getDefaultOrdScheme = function () {
    return "webanalyticsrollup:";
  };
  RelativeContributionChart.prototype.getColumnNames = function () {
    return this.columnNames;
  };
  RelativeContributionChart.prototype.getColorCodedColKeys = function () {
    return ['x'];
  };
  RelativeContributionChart.prototype.getDefaultSettings = function () {
    var defObj = AnalyticC3BaseChart.prototype.getDefaultSettings.apply(this, arguments);
    defObj.chartType = "pie";
    return defObj;
  };
  RelativeContributionChart.prototype.getXData = function (data) {
    return data.x;
  };
  RelativeContributionChart.prototype.getYData = function (data) {
    return data.val;
  };

  /**
   * Get Chart Params
   * @returns {*}
   */
  RelativeContributionChart.prototype.getTabParamType = function () {
    return baja.$("analytics:C3UxNoAxisWebChartParams");
  };

  /**
   * Get Settings Editor Type
   * @returns {*}
   */
  RelativeContributionChart.prototype.getSettingsEditorType = function () {
    return AnalyticRankingChartUXSettingsFE;
  };

  /**
   *
   * @param data
   * @param timeRange
   * @returns {*}
   */
  RelativeContributionChart.prototype.buildModel = function (data, timeRange) {
    return AnalyticC3BaseChart.prototype.buildModel.apply(this, arguments);
  };
  RelativeContributionChart.prototype.getName = function () {
    return "RelativeContributionChart";
  };
  return RelativeContributionChart;
});
