/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * Created by H129264 on 01-Jul-16.
 */
define(['baja!', 'bajaux/Widget', 'bajaux/mixin/subscriberMixIn', 'Promise', 'jquery', 'moment', 'underscore', 'd3', 'nmodule/analytics/rc/chart/base/AnalyticD3BaseChart', 'nmodule/analytics/rc/util/analyticsUtil', 'nmodule/analytics/rc/chart/utils/AnalyticsChartDragDropUtils', 'lex!baja,analytics', 'nmodule/analytics/rc/chart/fe/SpectrumUxWebChartSettingsFE', 'baja!analytics:SpectrumUxWebChartParams'], function (baja, Widget, subscriberMixIn, Promise, $, moment, _, d3, AnalyticD3BaseChart, analyticsUtil, AnalyticsChartDragDropUtils, lexicons, SpectrumUxWebChartSettingsFE, types) {
  "use strict";

  var lex = lexicons[1],
    SPECTRUM_TS_FORMAT = 'MMMM Do YYYY, HH:mm:ss';
  var MIN_TILE_HEIGHT = 1;
  var SpectrumChart = function SpectrumChart(params) {
    var that = this;
    AnalyticD3BaseChart.call(this, $.extend({}, params));
    that.xStep = 864e5; // 1 day buffer as xStep.
    that.yStep = 0; // Dynamic yStep will be assigned as needed.
    that.xScale = {};
    that.yScale = {};
    that.zScale = {}; // Axis fields.
    this.columnNames = ['Date', 'Value', lex.get("table.interpolationStatus.columnText")];
    that.columnKeys = ['jsDate', 'value', "interpolationStatus"];
  };
  SpectrumChart.prototype = Object.create(AnalyticD3BaseChart.prototype);
  SpectrumChart.prototype.constructor = SpectrumChart;

  /**
   * Draws a 2d Spectrum Chart with the AnalyticTrendArray as input.
   */
  SpectrumChart.prototype.reDraw = function (chartSettingsCollection) {
    var that = this;
    var model = chartSettingsCollection[0],
      analyticTrendArray = model.getAnalyticTrendArray();
    var stepInterval = analyticsUtil.decodeIntervalForStep(model.getInterval());
    that.yStep = stepInterval || 60; // Default it to an hour
    analyticTrendArray = $.map(analyticTrendArray, function (trendData) {
      var jsDate = trendData.timestamp;
      trendData.totalMins = jsDate.getHours() * 60 + jsDate.getMinutes();
      trendData.ts = moment(trendData.date.getJsDate()).startOf('day').toDate();
      trendData.jsDate = moment(jsDate).format(SPECTRUM_TS_FORMAT);
      return trendData;
    });
    analyticsUtil.updateByTailingDate(analyticTrendArray, that.yStep);
    that.draw(chartSettingsCollection);
  };

  /**
   *
   * @param that
   * @param d
   * @param model
   * @param dateString
   * @return {*}
   */
  function getInterpolTooltipTxt(that, d, model, dateString) {
    var ttipString;
    if (that.showInterpolationStatus) {
      ttipString = lex.get({
        key: "chart.spectrum.tooltip",
        args: [dateString, that.getFormattedTooltipValue(d.value), model.getUnit() && model.getUnit().getSymbol() !== "null" ? model.getUnit().getSymbol() : "", d.interpolationStatus]
      });
    } else {
      ttipString = lex.get({
        key: "chart.spectrum.withoutInterpoltooltip",
        args: [dateString, that.getFormattedTooltipValue(d.value), model.getUnit() && model.getUnit().getSymbol() !== "null" ? model.getUnit().getSymbol() : ""]
      });
    }
    return ttipString;
  }

  /**
   *
   * @param chartSettingsCollection
   * @returns {*}
   */
  SpectrumChart.prototype.draw = function (chartSettingsCollection) {
    var that = this,
      model = chartSettingsCollection[0],
      analyticTrendArray = model.getAnalyticTrendArray();
    that.series = [{
      name: model.getSeriesName(),
      ord: model.getOrd(),
      data: analyticTrendArray
    }];
    // Get the div for the tooltip
    var tooltipDiv = that.getToolTipDiv();
    // Define the boundaries of SVG.
    var svg = that.svg();
    // Define width & height for x and y axis.
    var width = that.availableWidth();
    var height = that.availableHeight();

    // Define scale for x, y & z axis.
    that.xScale = d3.time.scale().range([0, width * 0.9]);
    that.yScale = d3.scale.linear().range([height, 0]);

    // Utility method to get the value.
    var accessor = function accessor(val) {
      return val.value;
    };

    // Find the extent & mean to calcDomain that's used for Legend.
    var extent = d3.extent(analyticTrendArray, accessor);
    var mean = d3.mean(analyticTrendArray, accessor);
    that.calcDomain = [extent[0], mean, extent[1]];

    //NCCB-31148: Using linear scale to gracefully gradient.
    that.zScale = d3.scale.linear().domain([extent[0], mean, extent[1]]).range([model.getBrush(), model.getMidBrush(), model.getBrush2()]);

    // Define domain for x, y & z axis
    that.xScale.domain(d3.extent(analyticTrendArray, function (d) {
      return d.ts;
    }));

    // Domain is 0H to 24H
    that.yScale.domain([0, 1440]);

    // Set the axis extrapolation boundary on X & Y.
    that.xScale.domain([that.xScale.domain()[0], +that.xScale.domain()[1] + that.xStep]);

    // Display rectangle tiles.
    svg.append("g").attr('class', 'chartArea').selectAll(".tile").data(analyticTrendArray) // set the array of data to be plotted a rectangular tile.
    .enter().append("rect") // appends a rectangle
    .attr("class", "tile") // sets the class as tile
    .style("stroke-width", "1").style("stroke", function (d) {
      // fill the value over z axis
      return that.zScale(d.value);
    }).attr("x", function (d) {
      // sets the data that should be plotted over x as the timestamp.
      return that.xScale(d.ts);
    }).attr("y", function (d) {
      // sets the data that should be plotted over y as the total mins, yStep will help in where to start the tile.
      var yCoordinate;
      if (d.overFlowStep) {
        yCoordinate = that.yScale(d.overFlowStep);
      } else {
        yCoordinate = that.yScale(d.totalMins + that.yStep);
      }
      return yCoordinate < 0 ? 0 : yCoordinate;
    }).attr("width", that.xScale(that.xStep) - that.xScale(0)) // width of tile
    .attr("height", function (d) {
      var tileHeight, yCoordinate;
      if (d.overFlowStep) {
        tileHeight = that.yScale(0) - that.yScale(d.overFlowStep);
        yCoordinate = that.yScale(d.overFlowStep);
      } else {
        tileHeight = that.yScale(0) - that.yScale(that.yStep);
        yCoordinate = that.yScale(d.totalMins + that.yStep);
      }
      if (yCoordinate < 0) {
        tileHeight = tileHeight + yCoordinate;
      }
      tileHeight = tileHeight < MIN_TILE_HEIGHT ? MIN_TILE_HEIGHT : tileHeight;
      return tileHeight;
    }) // height of tile
    .on("mouseover", function (d) {
      // attach mouseover events
      tooltipDiv.transition().duration(200).style("opacity", 0.9);
      var mousePosition = d3.mouse(this);
      var dateString = moment(d.date.getJsDate()).format(SPECTRUM_TS_FORMAT);
      tooltipDiv.html(getInterpolTooltipTxt(that, d, model, dateString)).style("left", mousePosition[0] * 0.9 + "px").style("top", mousePosition[1] + 30 + "px");
    }).on("mouseout", function (d) {
      // attach mouseout events
      tooltipDiv.transition().duration(200).style("opacity", 0);
    }).style("fill", function (d) {
      // fill the value over z axis
      return that.zScale(d.value);
    });

    // debugValue to support the sanity testing, should be enabled through analyticService.
    if (that.isDebugEnabled) {
      d3.selectAll('rect.tile').attr('debugValue', function (d) {
        return d.ts + ";" + d.value + (["null", undefined].contains(model.getUnit().toString()) ? "" : ";" + model.getUnit().toString());
      });
    }
    var axis = that.appendYAxis(that.appendXAxis(that.appendLegend(svg, width, model, height), width, height, model), height, model);
    that.applyAxisTickTextSize();
    return Promise.resolve(axis);
  };

  /**
   * Appends legend to the svg.
   * @param svg
   * @param width
   * @returns {*}
   */
  SpectrumChart.prototype.appendLegend = function (svg, width, model, height) {
    var that = this;
    var zScale = that.zScale;
    var textSize = that.getTabConfigDataModel().getFontSize();
    // Define scales to dynamically decide on width & height of colors and text.
    var widthScale = d3.scale.linear().domain([0, 1500]).range([5, 80]);
    var heightScale = d3.scale.linear().domain([0, 1000]).range([4, 15]);
    //      var heightScaleForEm = d3.scale.linear().domain([100, 1000]).range([0.01, 0.35]);
    // Add a legend for the color values.
    var ticks = [];
    // Shows 10 values on the legend, using the input calcDomain, with 2 pointer decimal precision.
    ticks.push.apply(ticks, d3.scale.linear().domain(that.calcDomain).nice(10).nice(10).ticks(10).reverse());
    // ticks.push.apply(ticks, d3.scale.linear().domain(that.calcDomain).ticks(10).reverse());

    // If ticks are empty then add just 1 tick with value.
    if (_.isEmpty(ticks)) {
      ticks.push(that.calcDomain[0]);
    }
    var calcWidth = widthScale(width);
    var startWidth = width * 0.97 - calcWidth;
    var calcHeight = heightScale(height);
    var legend = d3.select("svg").append("g").attr("class", "legendArea").attr("transform", "translate(" + that.chartMargins().left + ",0)").selectAll(".legend").data(ticks).enter().append("g").attr("class", "legend").attr("transform", function (d, i) {
      return "translate(" + (startWidth - calcWidth * i) + "," + 0 + ")";
    });

    // Add a rect for the legend
    legend.append("rect").attr("width", calcWidth).attr("height", calcHeight).style("stroke-width", "1").style("stroke", "rgb(0,0,0)").style("fill", zScale);

    //Define a text for the legend
    legend.append("text").attr("x", 0).attr("y", 2 * calcHeight + 2).style("font-size", textSize + "px").text(function (e) {
      return analyticsUtil.standardizeInput(e);
    });

    // var valString = lex.get("chart.spectrum.legend.value");
    // var valueText = (!(model.getUnit() && model.getUnit().getSymbol() !== "null")) ? valString : valString + " (" +
    //   model.getUnit().getSymbol() + ")";
    // // Add a text for legend
    // svg.append("text")
    //   .attr("class", "label")
    //   .attr("x", width * 0.95)
    //   .attr("y", 10)
    //   .style("font-size", textSize + "px")
    //   .attr("dy", heightScaleForEm(height).toFixed(2) + "em")
    //   .text(valueText);
    return svg;
  };

  /**
   * Appends x axis to the svg.
   * @param svg
   * @param width
   * @param height
   * @returns {*}
   */
  SpectrumChart.prototype.appendXAxis = function (svg, width, height, model) {
    var that = this;
    // Add an x-axis with label.
    that.initXAxis();
    return svg;
  };
  SpectrumChart.prototype.getXAxis = function () {
    var that = this,
      xScale = that.xScale,
      tickFormat,
      intervals,
      axis,
      divideBy,
      dayCount = Math.abs((xScale.domain()[1].getTime() - xScale.domain()[0].getTime()) / 864e5);
    if (dayCount <= 14) {
      tickFormat = d3.time.days;
      divideBy = 1;
    } else if (dayCount > 14 && dayCount <= 105) {
      tickFormat = d3.time.weeks;
      divideBy = 7;
    } else if (dayCount > 105 && dayCount <= 366) {
      tickFormat = d3.time.months;
      divideBy = 30;
    } else {
      tickFormat = d3.time.years;
      divideBy = 365;
    }
    intervals = Math.round(dayCount / divideBy);
    if (intervals > this.getTickCountForXAxis()) {
      // ticks(d3.time.years, 1) will display ticks with 1 year gap.
      // Ceiled it, to get the best chance to get less number of ticks.
      axis = d3.svg.axis().scale(xScale).ticks(tickFormat, Math.ceil(Math.abs(intervals / this.getTickCountForXAxis())));
    } else {
      axis = d3.svg.axis().scale(xScale).ticks(tickFormat);
    }
    axis.tickFormat(analyticsUtil.formatDate).orient("bottom");
    return axis;
  };

  /**
   * Appends y axis to svg.
   * @param svg
   * @param height
   * @returns {*}
   */
  SpectrumChart.prototype.appendYAxis = function (svg, height, model) {
    var that = this;
    // Add a y-axis with label.
    that.initYAxis();
    return svg;
  };
  SpectrumChart.prototype.getYAxis = function () {
    var that = this;
    var yScale = that.yScale;
    // Default the step count to the 30 minutes, if user selects interval less than 30 minutes.
    var stepScale = d3.scale.threshold().domain([100, 250, 350, 500]).range([30, 60, 120, 180, 240].reverse());
    var axis = that.buildYAxis(yScale).tickValues(d3.range(0, yScale.domain()[1], stepScale(that.availableHeight()))).tickFormat(analyticsUtil.formatMinutes);
    return axis;
  };

  /**
   * This method should be overridden by subclasses to return respective chart types.
   * The default implementation retruns "none".
   * @returns {string}
   */
  SpectrumChart.prototype.getSupportedExportTypes = function () {
    return ["achart", "csv"];
  };

  /**
   * This needs a clean slate
   * @returns {boolean}
   */
  SpectrumChart.prototype.needsCleanSlate = function () {
    return true;
  };

  /**
   * Get Chart Params
   * @returns {*}
   */
  SpectrumChart.prototype.getTabParamType = function () {
    return baja.$("analytics:SpectrumUxWebChartParams");
  };

  /**
   * Get Settings Editor Type
   * @returns {*}
   */
  SpectrumChart.prototype.getSettingsEditorType = function () {
    return SpectrumUxWebChartSettingsFE;
  };

  /**
   * Build Chart Model
   * @param data
   * @param timeRange
   * @returns {boolean}
   */
  SpectrumChart.prototype.buildModel = function (data, timeRange) {
    var model = AnalyticD3BaseChart.prototype.buildModel(data, timeRange);
    model.setBrush2(data.getBrush2());
    return model;
  };

  /**
   * Build Chart Settings
   * @param chartSettingsColl
   * @param setting
   * @returns {*}
   */
  SpectrumChart.prototype.buildSettings = function (chartSettingsColl, setting) {
    var settings = AnalyticD3BaseChart.prototype.buildSettings(chartSettingsColl, setting);
    settings.setBrush2(chartSettingsColl.getBrush2());
    return settings;
  };

  /**
   *
   * @returns {number}
   */
  SpectrumChart.prototype.getLabelPositionX = function () {
    return this.availableWidth() / 2.5;
  };

  /**
   * Get the chart margins
   * @type {{top: number, right: number, bottom: number, left: number}}
   */
  SpectrumChart.prototype.chartMargins = function () {
    return {
      top: 30,
      right: 0,
      bottom: 50,
      left: 40
    };
  };
  SpectrumChart.prototype.getXAxisLabel = function (text) {
    return "Date";
  };
  SpectrumChart.prototype.getYAxisLabel = function (text) {
    return "Hours";
  };
  SpectrumChart.prototype.isMultiBindingSupported = function () {
    return false;
  };
  SpectrumChart.prototype.isClrToBeDisplayedInRpt = function () {
    return true;
  };
  SpectrumChart.prototype.getColorCodedColKeys = function () {
    return ['value'];
  };
  SpectrumChart.prototype.getName = function () {
    return "SpectrumChart";
  };
  SpectrumChart.prototype.doDestroy = function () {
    this.jq().find('.chartArea').removeData();
    this.jq().find('.chartContainer').removeData();
    return Promise.all([d3.select('.chartArea').selectAll('svg').remove(), d3.select('.chartContainer').selectAll('svg').remove()]);
  };
  return SpectrumChart;
});
