/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/schedule/rc/baja/wb/TriggerScheduler
 */
define(['bajaux/mixin/subscriberMixIn', 'bajaux/util/SaveCommand', 'bajaux/mixin/responsiveMixIn', 'nmodule/js/rc/log/Log', 'nmodule/schedule/rc/baja/fe/ActiveDatesCalendar', 'nmodule/schedule/rc/baja/fe/TriggerDayEditor', 'nmodule/schedule/rc/fe/CalendarEventsEditor', 'nmodule/schedule/rc/servlets/ssc', 'nmodule/schedule/rc/util/scheduleUtils', 'nmodule/webEditors/rc/fe/CompositeEditor', 'nmodule/webEditors/rc/fe/config/CompositeBuilder', 'Promise'], function (subscribable, SaveCommand, responsiveMixIn, Log, ActiveDatesCalendar, TriggerDayEditor, CalendarEventsEditor, ssc, scheduleUtils, CompositeEditor, CompositeBuilder, Promise) {
  'use strict';

  var tpl = function tpl(ed) {
    return "\n    <div class=\"row header\">\n        <div class=\"js-activeDates\"></div>\n    </div>\n    <div class=\"row footer\">\n      <div class=\"js-dates\"></div>\n      <div class=\"js-times\"></div>\n    </div>";
  };

  /**
   * Editor for a `schedule:TriggerSchedule`.
   *
   * @class
   * @alias module:nmodule/schedule/rc/baja/wb/TriggerScheduler
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   */
  var TriggerScheduler = function TriggerScheduler() {
    CompositeEditor.apply(this, arguments);
    this.getCommandGroup().add(new SaveCommand());
    subscribable(this);
    responsiveMixIn(this, {
      'trigger-scheduler-lg': {
        minWidth: 535
      }
    });
  };
  TriggerScheduler.prototype = Object.create(CompositeEditor.prototype);
  TriggerScheduler.prototype.constructor = TriggerScheduler;

  /**
   * @returns {module:nmodule/webEditors/rc/fe/config/CompositeBuilder}
   */
  TriggerScheduler.prototype.makeBuilder = function () {
    var _this = this;
    var builder = new CompositeBuilder();
    builder.getKeys = function () {
      return ['dates', 'times', 'activeDates'];
    };
    builder.getDomFor = function (key) {
      return _this.jq().find(".js-".concat(key));
    };
    builder.getConfigFor = function (key) {
      switch (key) {
        case 'activeDates':
          return {
            properties: {
              refBase: _this.$refBase,
              sourceOrd: _this.$sourceOrd
            },
            type: ActiveDatesCalendar
          };
        case 'dates':
          return {
            type: CalendarEventsEditor,
            properties: {
              includeReference: _this.$includeReference(),
              sourceOrd: _this.$sourceOrd
            }
          };
        case 'times':
          return {
            type: TriggerDayEditor
          };
      }
    };
    builder.getValueFor = function (key) {
      switch (key) {
        case 'activeDates':
          return _this.$snapshot.get('dates');
        default:
          return _this.$snapshot.get(key);
      }
    };
    return builder;
  };

  /**
   * @param {jQuery} dom
   * @returns {Promise}
   */
  TriggerScheduler.prototype.doInitialize = function (dom) {
    dom.addClass('TriggerScheduler').html(tpl(this));
    return CompositeEditor.prototype.doInitialize.apply(this, arguments);
  };

  /**
   * Return false not include ScheduleReference as an option when
   * adding a new Event. This prevents the TriggerScheduler from creating
   * a reference to itself.
   * @returns {boolean}
   * @private
   */
  TriggerScheduler.prototype.$includeReference = function () {
    return false;
  };
  TriggerScheduler.prototype.doLayout = function () {
    var width = this.jq().find('.js-times').outerWidth(true);
    if (this.jq().hasClass('trigger-scheduler-lg')) {
      this.jq().find('.js-dates').css('padding-right', "".concat(width, "px"));
      this.jq().find('.row.header').css('padding-right', "".concat(width, "px"));
    } else {
      this.jq().find('.js-dates').css('padding-right', '');
      this.jq().find('.row.header').css('padding-right', '');
    }
    return CompositeEditor.prototype.doLayout.apply(this, arguments);
  };

  /**
   * @param {baja.Component} sched a `schedule:TriggerSchedule`
   * @returns {Promise}
   */
  TriggerScheduler.prototype.doLoad = function (sched) {
    var _arguments = arguments,
      _this2 = this;
    var navOrd = sched.getNavOrd();
    return ssc.loadScheduleInfo(navOrd).then(function (_ref) {
      var snapshot = _ref.snapshot,
        refBase = _ref.refBase,
        options = _ref.options;
      _this2.$refBase = refBase;
      _this2.$snapshot = snapshot;
      _this2.$sourceOrd = sched.getNavOrd();
      _this2.$options = options;
      return CompositeEditor.prototype.doLoad.apply(_this2, _arguments).then(function () {
        _this2.$getDatesEditor().on('selectedEventChanged', function (event) {
          var sched = event || _this2.$getDatesEditor().value();
          _this2.$getActiveDatesCalendar().load(sched)["catch"](Log.error);
        });
        if (options.readonly) {
          return _this2.setReadonly(true);
        }
      });
    });
  };

  /**
   * Save changes to the `TriggerSchedule`.
   * @returns {Promise}
   */
  TriggerScheduler.prototype.doSave = function () {
    var _this3 = this;
    return this.$getDatesEditor().save().then(function () {
      return ssc.save(_this3.value().getNavOrd(), _this3.$snapshot);
    });
  };
  TriggerScheduler.prototype.doDestroy = function () {
    this.jq().removeClass('TriggerScheduler');
    return CompositeEditor.prototype.doDestroy.apply(this, arguments);
  };

  /**
   * @private
   * @returns {module:nmodule/schedule/rc/baja/fe/ActiveDatesCalendar}
   */
  TriggerScheduler.prototype.$getActiveDatesCalendar = function () {
    return this.getBuilder().getEditorFor('activeDates');
  };

  /**
   * @private
   * @returns {module:nmodule/schedule/rc/fe/CalendarEventsEditor}
   */
  TriggerScheduler.prototype.$getDatesEditor = function () {
    return this.getBuilder().getEditorFor('dates');
  };

  /**
   * @private
   * @returns {module:nmodule/schedule/rc/baja/fe/TriggerDayEditor}
   */
  TriggerScheduler.prototype.$getTimesEditor = function () {
    return this.getBuilder().getEditorFor('times');
  };
  return TriggerScheduler;
});
