/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/schedule/rc/fe/WeekEditor
 */
define(['bajaux/commands/Command', 'bajaux/events', 'bajaux/mixin/responsiveMixIn', 'jquery', 'Promise', 'underscore', 'nmodule/schedule/rc/fe/DayEditor', 'nmodule/schedule/rc/fe/templates', 'nmodule/schedule/rc/model/WeekSchedule', 'nmodule/webEditors/rc/fe/config/CompositeBuilder', 'nmodule/webEditors/rc/fe/CompositeEditor', 'nmodule/webEditors/rc/mixin/zoomMixIn'], function (Command, events, responsiveMixIn, $, Promise, _, DayEditor, templates, WeekSchedule, CompositeBuilder, CompositeEditor, zoomMixIn) {
  'use strict';

  var MODIFY_EVENT = events.MODIFY_EVENT,
    scheduleLabels = templates.scheduleLabels,
    DAYS_OF_WEEK = ['sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'],
    WEEKDAYS = DAYS_OF_WEEK.slice(1, 6);
  var tpl = function tpl(ed) {
    return "\n    <div class=\"weekEditorOuter\">\n      <div class=\"weekEditorContainer\">\n        ".concat(scheduleLabels(ed.$getTimeLabels()), "\n        <div class=\"weekdays\"></div>\n      </div>\n    </div>");
  };

  ////////////////////////////////////////////////////////////////
  // Commands
  ////////////////////////////////////////////////////////////////

  var ApplyMFCommand = function ApplyMFCommand(ed) {
    Command.call(this, {
      module: 'schedule',
      lex: 'commands.applyMF',
      func: function func(scheduleBlocks) {
        var builder = ed.getBuilder();
        return Promise.all(WEEKDAYS.map(function (weekday) {
          return builder.getEditorFor(weekday).$getApplyCommand().invoke(scheduleBlocks);
        }));
      }
    });
  };
  ApplyMFCommand.prototype = Object.create(Command.prototype);
  ApplyMFCommand.prototype.constructor = ApplyMFCommand;
  var ClearWeekCommand = function ClearWeekCommand(ed) {
    Command.call(this, {
      module: 'schedule',
      lex: 'commands.clearWeek',
      func: function func() {
        return Promise.all(ed.$getDayEditors().map(function (dayEditor) {
          return dayEditor.$getClearCommand().invoke();
        }));
      }
    });
  };
  ClearWeekCommand.prototype = Object.create(Command.prototype);
  ClearWeekCommand.prototype.constructor = ClearWeekCommand;

  ////////////////////////////////////////////////////////////////
  // Definition
  ////////////////////////////////////////////////////////////////

  /**
   * A `WeekEditor` shows a weekly view containing "draggy bar" editors for up
   * to seven days of the week.
   *
   * It supports the following bajaux Properties:
   *
   * - `firstDayOfWeek`: configured first day of week (full week name, English,
   *   lowercase). default: sunday
   * - `timeLabels`: an array of the time labels showing the times at the most
   *   prominent horizontal lines. default: 3 AM, 6 AM, 9 AM, 12 PM, 3 PM, 6 PM,
   *   9 PM
   *
   * @class
   * @alias module:nmodule/schedule/rc/fe/WeekEditor
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   */
  var WeekEditor = function WeekEditor() {
    CompositeEditor.apply(this, arguments);
    this.getCommandGroup().add(new ApplyMFCommand(this), new ClearWeekCommand(this));
    responsiveMixIn(this, {
      'week-editor-sm': {
        maxWidth: 758
      },
      'week-editor-xs': {
        maxWidth: 430
      }
    });
    zoomMixIn(this, {
      zoomTargets: ['.weekEditorContainer']
    });
  };
  WeekEditor.prototype = Object.create(CompositeEditor.prototype);
  WeekEditor.prototype.constructor = WeekEditor;
  WeekEditor.prototype.$getApplyMFCommand = function () {
    return this.getCommandGroup().get(0);
  };
  WeekEditor.prototype.$getClearWeekCommand = function () {
    return this.getCommandGroup().get(1);
  };

  ////////////////////////////////////////////////////////////////
  // bajaux implementation
  ////////////////////////////////////////////////////////////////

  /**
   * Arm event handlers to propagate block selection and effective value changes
   * across days.
   *
   * @param {jQuery} dom
   */
  WeekEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.addClass('WeekEditor').html(tpl(this));
    dom.on('selectionChange', '.DayEditor', function (e, ed, block) {
      that.$getDayEditors().forEach(function (dayEd) {
        dayEd.$setSelectedBlock(block);
      });
    });
    dom.on(MODIFY_EVENT, '.DayEditor', function (e, ed) {
      that.setModified(true);
      var effectiveValue = ed.$getDefaultEffectiveValue();
      that.$getDayEditors().forEach(function (dayEd) {
        dayEd.$setDefaultEffectiveValue(effectiveValue);
      });
    });
  };

  /**
   * The builder for a `WeekEditor` maps the days in the week to individual
   * day editors.
   * @returns {module:nmodule/webEditors/rc/fe/config/CompositeBuilder}
   */
  WeekEditor.prototype.makeBuilder = function () {
    var that = this,
      builder = new CompositeBuilder();
    builder.$getDaySchedules = function () {
      return this.getDataSource().getDaySchedules();
    };
    builder.getKeys = function () {
      var daySchedules = this.$getDaySchedules(),
        weekdays = that.$getConfiguredWeekdays();
      return _.compact(weekdays.map(byDayOfWeek.bind(null, daySchedules))).map(function (ds) {
        return ds.getDayOfWeek();
      });
    };
    builder.getValueFor = function (key) {
      return _.find(this.$getDaySchedules(), function (ds) {
        return ds.getDayOfWeek() === key;
      });
    };
    builder.getConfigFor = function () {
      return {
        properties: that.properties().toValueMap(),
        type: DayEditor
      };
    };
    builder.getDomFor = function () {
      return $('<div/>').appendTo(that.$getWeekdaysElement());
    };
    return builder;
  };

  /**
   * @returns {Promise.<module:nmodule/schedule/rc/model/WeekSchedule>}
   */
  WeekEditor.prototype.doRead = function () {
    return this.$getDayEditors().readAll().then(function (daySchedules) {
      return new WeekSchedule({
        daySchedules: daySchedules
      });
    });
  };

  /**
   * @returns {Promise}
   */
  WeekEditor.prototype.doDestroy = function () {
    this.jq().removeClass('WeekEditor');
    return CompositeEditor.prototype.doDestroy.apply(this, arguments);
  };

  /**
   * Get the element that holds the `DayEditor` for each edited day.
   * @private
   * @returns {jQuery}
   */
  WeekEditor.prototype.$getWeekdaysElement = function () {
    return this.jq().find('.weekdays');
  };
  WeekEditor.prototype.$getConfiguredWeekdays = function () {
    var firstDayOfWeek = this.$getFirstDayOfWeek(),
      index = _.indexOf(DAYS_OF_WEEK, firstDayOfWeek);
    return DAYS_OF_WEEK.slice(index).concat(DAYS_OF_WEEK.slice(0, index));
  };
  WeekEditor.prototype.$getDayEditors = function () {
    return this.getChildWidgets(this.$getWeekdaysElement());
  };
  WeekEditor.prototype.$getFirstDayOfWeek = function () {
    return this.properties().getValue('firstDayOfWeek') || 'sunday';
  };
  WeekEditor.prototype.$getTimeLabels = function () {
    var timeLabels = this.properties().getValue('timeLabels') || ['3 AM', '6 AM', '9 AM', '12 PM', '3 PM', '6 PM', '9 PM'];
    if (timeLabels.length !== 7) {
      throw new Error('timeLabels length must be 7');
    }
    return timeLabels;
  };
  function byDayOfWeek(daySchedules, dayOfWeek) {
    return _.find(daySchedules, function (daySchedule) {
      return daySchedule.getDayOfWeek() === dayOfWeek;
    });
  }
  return WeekEditor;
});
